/* v1.1
 *
 * shields.c:  Ship shield code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "class.h"
#include "object.h"
#include "damage.h"
#include "shields.h"
#include "smisc.h"
#include "events.h"

const char *shdFullName(const TAG *object, int shield)
{
    const char *full_names[NUM_SHIELD_CONFIGS][NUM_SHIELDS] =
    {{"Fore", "Aft", "Port", "Starboard", "", "" },
     {"Fore", "Aft", "Port", "Starboard", "Dorsal", "Ventral"},
     {"Fore", "Aft", "Port", "Starboard", "Dorsal", "Ventral"},
     {"Dorsal", "Ventral", "Port", "Starboard", "", ""},
     {"Port-Dorsal", "Starboard-Dorsal", "Port-Ventral", 
      "Starboard-Ventral", "", ""},
     {"", "", "", "", "", ""},
     {"Main", "", "", "", "", ""}};
    
    /* Sanity check */
    if (shield >= NUM_SHIELDS) {
	log_space("WARNING: shield number out of range (%d) for #%d",
		  shield, object->rec->db_obj);
	return "";
    }
    
    if (Ship(object))
	return full_names[F_INT(object->shipdata->rec, shield_config)][shield];
    
    return "";
}

const char *shdShortName(const TAG *object, int shield)
{
    const char *short_names[NUM_SHIELD_CONFIGS][NUM_SHIELDS] =
    {{"Fore", "Aft", "Port", "Stbd", "", "" },
     {"Fore", "Aft", "Port", "Stbd", "Drsl", "Vtrl"},
     {"Fore", "Aft", "Port", "Stbd", "Drsl", "Vtrl"},
     {"Drsl", "Vtrl", "Port", "Stbd", "", ""},
     {"PtDl", "SbDl", "PtVl", "SbVl", "", ""},
     {"", "", "", "", "", ""},
     {"Main", "", "", "", "", ""}};
    
    if (Ship(object))
	return short_names[F_INT(object->shipdata->rec, shield_config)][shield];
    
    return "";
}

const char *shdCharName(const TAG *object, int shield)
{
    const char *short_names[NUM_SHIELD_CONFIGS][NUM_SHIELDS] =
    {{ "^",  "v",  "<",  ">",  "",   "" },
     { "^",  "v",  "<",  ">",  "*",  "O"},
     { "^",  "v",  "<",  ">",  "*",  "O"},
     { "*",  "O",  "<",  ">",  "",   "" },
     { "<^", "^>", "<v", "v>", "",   ""},
     { "",   "",   "",   "",   "",   ""},
     { "O",  "",   "",   "",   "",   ""}};
    
    if (Ship(object))
	return short_names[F_INT(object->shipdata->rec, shield_config)][shield];
    
    return "";
}

const char *shdConfigName(const TAG *object)
{
    const char *config_names[NUM_SHIELD_CONFIGS] =
    { "Type 1 Array (4/PSAF-60)",
      "Type 2 Array (6/PSDVAF-60)",
      "Type 3 Array (6/PSDVAF-45)",
      "Type 4 Array (4/PSVD)",
      "Type 5 Array (4/PSVD-45)",
      "None",
      "Type 7 Array (Single Shield)" };
    
    return config_names[F_INT(object->shipdata->rec, shield_config)];
}

void shdAllocCheck(TAG *object)
{
    SHIP *ship = object->shipdata;
    int i;
    float allocation = 0.0;
    dbref shdofficer;
    
    shdofficer = 0; /* Not used */
    
    for (i=0; i < NUM_SHIELDS; i++) 	
	allocation += ship->shield_alloc[i];	
    
    if ((ship->cloak_status == CLOAK_ON) || (ship->cloak_status == CLOAKING))
	allocation += F_INT(ship->rec, cloak_cost);
    
    if (allocation > ship->alloc_shields) {
	MFNotify(ship, -1, CONS_SHD, CONS_ACTIVE, "[Shield allocation "
		 "cut:");
	shdAllocate(object, shdofficer, ship->shield_alloc);
    }
    else 
	MFNotify(ship, -1, CONS_SHD, CONS_ACTIVE,
		 "[Energy allocation cut.  New allocation is "
		 "%d.  No shortfall]", ship->alloc_shields);
    
    return;
}


void shdAllocate(TAG *object, dbref player, int *shields)
{
    SHIP *ship = object->shipdata;
    int shield;
    float total_allocation;
    float power_ratio;
    
    /* Set any negative allocations to zero */
    for (shield = 0, total_allocation = 0; shield < NUM_SHIELDS; shield ++) {
	if (shields[shield] < 0)
	    shields[shield] = 0;
	
	total_allocation += shields[shield];
    }
    
    /* Check for overallocation */
    if (total_allocation > ship->alloc_shields) {
	power_ratio = (ship->alloc_shields) / total_allocation;
	
	for (shield = 0; shield < NUM_SHIELDS; shield ++)
	    shields[shield] = power_ratio * shields[shield];
    }
    
    /* Assign the new values */
    for (shield = 0; shield < NUM_SHIELDS; shield ++)
	ship->shield_alloc[shield] = shields[shield];
    
    /* check for cloak power */
    if ((ship->cloak_status == CLOAK_ON) || (ship->cloak_status == CLOAKING)) {
	if ((ship->alloc_shields - total_allocation) < 
	    F_INT(ship->rec, cloak_cost)) {
	    if (ship->cloak_status == CLOAKING)
		MFNotify(ship, player, CONS_SHD, CONS_ACTIVE,
			 "Insufficient energy to continue "
			 "establishing cloak.");
	    else	
		MFNotify(ship, player, CONS_SHD, CONS_ACTIVE,
			 "Insufficient energy to maintain cloak.");
	    
	    MFNotify(ship, player, CONS_SHD, CONS_ACTIVE,
		     "Cloak requires %d unallocated power to shields.", 
		     F_INT(ship->rec, cloak_cost));
	    shdCloakOff(object, player);
	}
    }
    
    /* Notify the player */
    MFNotify(ship, player, CONS_SHD, CONS_ACTIVE,
	     "Total available power: %d\nAllocations:", ship->alloc_shields);
    
    switch(F_INT(ship->rec, shield_config)) {
    case SHIELD_CONFIG_ONE:
	MFNotify(ship, player, CONS_SHD, CONS_ACTIVE,
		 "Shields:  %s:%4d\n", 
		 shdShortName(object,0), shields[0]);
	break;

    case SHIELD_CONFIG_ORIG6:
    case SHIELD_CONFIG_CUBE:
	MFNotify(ship, player, CONS_SHD, CONS_ACTIVE,
		 "Shields:  %s:%4d  %s:%4d  %s:%4d"
		 "  %s:%4d  %s:%4d  %s:%4d\n", 
		 shdShortName(object,0), shields[0],
		 shdShortName(object,1), shields[1],
		 shdShortName(object,2), shields[2],
		 shdShortName(object,3), shields[3],
		 shdShortName(object,4), shields[4],
		 shdShortName(object,5), shields[5]);
	break;
	
    case SHIELD_CONFIG_ORIG4:
    case SHIELD_CONFIG_ORANGE1:
    case SHIELD_CONFIG_ORANGE2:
    default:
	MFNotify(ship, player, CONS_SHD, CONS_ACTIVE,
		 "Shields:  %s:%4d  %s:%4d  %s:%4d"
		 "  %s:%4d\n", 
		 shdShortName(object,0), shields[0],
		 shdShortName(object,1), shields[1],
		 shdShortName(object,2), shields[2],
		 shdShortName(object,3), shields[3]);
	break;
	
    }
}

void shdMaintainShields(TAG *object)
{
#ifdef ENABLE_SHIELD_CLASSES
    /* FIX ME */
#else
    SHIP *ship = object->shipdata;
    int shield_additions[NUM_SHIELDS], old_shield_level[NUM_SHIELDS], i;
    
    /* Store previous values, remove overload energy, and bleed */
    for (i = 0; i < NUM_SHIELDS; i++) {
	old_shield_level[i] = ship->shield_level[i];
	
	if (ship->shield_level[i] > ship->shield_max[i])
	    ship->shield_level[i] = ship->shield_max[i];
	
	ship->shield_level[i] *= (ship->shield_status[i]==SHLD_READY) 
	    ?  SHLD_READY_BLEED : SHLD_UP_BLEED;
	
	/* Calculate shield additions */	
	shield_additions[i] = Min(ship->shield_alloc[i],
				  ship->shield_max_charge[i]);
	
	/* now add any allocated points if the shield is working and
	 * truncate overloads at overload max and update action 
	 * registers
	 */
	
	if (ship->shield_status[i] != SHLD_DAMAGED)
	    ship->shield_level[i] += (int)(shield_additions[i] *
					   F_FLOAT(ship->rec, shield_factor));
	
	if ((Cloaked(object) || ship->cloak_status == CLOAKING) &&
	    ship->shield_level[i] > old_shield_level[i])
	    ship->shield_level[i] = old_shield_level[i];
	
	if (ship->shield_level[i] == old_shield_level[i])
	    ship->shield_action[i] = SHLD_STABLE;
	else
	    ship->shield_action[i] =
		(ship->shield_level[i] > old_shield_level[i]) ?
		SHLD_CHARGING : SHLD_FALLING;
	
	if (ship->shield_level[i] > ship->shield_max[i]) {
	    ship->shield_action[i] = SHLD_OVERLOADED;
	    
	    if (ship->shield_status[i] != SHLD_UP)
		ship->shield_level[i] = ship->shield_max[i];
	}
	
    }

#endif
    
    return;
}

void shdCloakOn(TAG *object, dbref player)
{
	SHIP *ship = object->shipdata;
	int i, total_allocation = 0;

	if (Naked(ship)) {
		Notify(player, "Cloak inoperable.");
		return;
	}

	/* make sure we're even allowed to cloak */
	switch (ship->damage[SYSTEM_CLOAK].status) {
	  case 'x':
	  case 'X':
		Notify(player, "This ship has no cloaking device.");
		return;

	  case '7':
	  case '9':
		Notify(player, "The cloaking device is damaged and may not be "
		  "used until repaired.");
		return;
	}

	/* make sure it isn't already on */
	if (Cloaked(object) || (ship->cloak_status==CLOAKING)) {
		Notify(player, "Cloak is already engaged.");
		return;
	}

#ifdef ENABLE_SHIELD_CLASSES
	/* FIX ME */
#else
	/* check for shields */
	for (i = 0; i < ship->shield_number; i++) {
		if (ship->shield_status[i] == SHLD_DAMAGED) {
			Notify(player, "All shields must be undamaged to use "
			  "cloak.");
			return;
		}
		total_allocation += ship->shield_alloc[i];
	}

	/* Make sure the shield allocation is enough to run the cloak */

	if (ship->alloc_shields - total_allocation < 
	    F_INT(ship->rec, cloak_cost)) {
		Notify(player, "Insufficient energy to engage cloak.");
		FNotify(player, "Cloak requires %d unallocated power to "
			"shields.", F_INT(ship->rec, cloak_cost));
		return;
	}
#endif

	/* Time to cloak.  Set the cloak status first so that the shield  *
	 * functions will know to suppress the messages.                  */

	ship->cloak_status = CLOAKING;
	ship->time_to_cloak = 2;

#ifdef ENABLE_SHIELD_CLASSES
	/* FIX ME */
#else
	for (i = 0; i < ship->shield_number; i++) 
		shdLowerShield(object, player, i);
#endif

	MFNotify(ship, player, CONS_SHD, CONS_ACTIVE, "[Engaging cloaking "
		 "device]");

	/* notify anyone who can see us */
	evBroadcast(object, NULL, "is cloaking", EVENT_CLOAK_OTHER);

	return;
}

void shdCloakOff(TAG *object, dbref player)
{
	SHIP *ship = object->shipdata;

	/* switch on the status */
	switch(ship->damage[SYSTEM_CLOAK].status) {
	  case 'x':
	  case 'X':
		Notify(player, "This ship has no cloaking device.");
		return;

	  case '7':
	  case '9':
		Notify(player, "The cloaking device is damaged and need"
		  " not be disengaged.");
		return;
	}

	/* make sure it isn't already off */
	if (!Cloaked(object) || (ship->cloak_status==DECLOAKING)) {
		Notify(player, "Cloak is already disengaged.");
		return;
	}

	/* terrific.  We're cloaked.  Change the status register */
	ship->cloak_status = DECLOAKING;
	object->tagflags &= ~CLOAKED;
	ship->time_to_cloak = 2;

	/* notify anyone who can see us */
	evBroadcast(object, NULL, "is decloaking", EVENT_DECLOAK_OTHER);

	MFNotify(ship, player, CONS_SHD, CONS_ACTIVE, "[Disengaging cloaking "
		 "device]");
	return;
}

void shdRaiseShield(TAG *object, dbref player, int shield)
{
	SHIP *ship = object->shipdata;

	if (Naked(ship)) {
		Notify(player, "Shields inoperable.");
		return;
	}

	/* don't raise if cloaked */
	if (Cloaked(object) || ship->cloak_status == CLOAKING) {
		Notify(player, "Shields cannot be raised while "
		  "cloaking device is engaged.");
		return;
	}

#ifdef ENABLE_SHIELD_CLASSES
	/* FIX ME */
#else
	if (shield >= ship->shield_number) {
		FNotify(player, "Invalid shield (%d) for this configuration.",
			shield);
		return;
	}

	switch (ship->shield_status[shield]) {
		case SHLD_READY:
			ship->shield_status[shield] = SHLD_UP;
			MFNotify(ship, player, CONS_SHD, CONS_ACTIVE, "[%s "
				 "shield raised]",
				shdFullName(object, shield));
			break;

		case SHLD_UP:
			FNotify(player, "%s shield is already up.",
				shdFullName(object, shield));
			break;

		case SHLD_DAMAGED:
			FNotify(player, "%s shield is damaged and "
				"may not be raised or lowered.",
				shdFullName(object, shield));
			break;
	}
#endif

	return;
}

void shdLowerShield(TAG *object, dbref player, int shield)
{
	SHIP *ship = object->shipdata;

	if (Naked(ship)) {
		Notify(player, "Shields inoperable.");
		return;
	}

#ifdef ENABLE_SHIELD_CLASSES
	/* FIX ME */
#else
	if (shield >= ship->shield_number) {
		FNotify(player, "Invalid shield (%d) for this configuration.",
			shield);
		return;
	}

	switch (ship->shield_status[shield]) {
		case SHLD_READY:
			if (!Cloaked(object) && ship->cloak_status != CLOAKING)
				FNotify(player, "%s shield is already down.",
				shdFullName(object, shield));
			break;

		case SHLD_UP:
			ship->shield_status[shield] = SHLD_READY;
			if (!Cloaked(object) && ship->cloak_status != CLOAKING)
				MFNotify(ship, player, CONS_SHD, CONS_ACTIVE,
					"[%s shield lowered]", 
					 shdFullName(object, shield));
			break;

		case SHLD_DAMAGED:
			FNotify(player, "%s shield is damaged and "
				"may not be raised or lowered.",
				shdFullName(object, shield));
			break;
	}
#endif

	return;
}
