/* v0.9
 *
 * sensors.c:  Sensor code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "sensors.h"
#include "damage.h"
#include "smisc.h"
#include "shields.h"
#include "events.h"
#include "tactical.h"
#include "nav.h"

#define Shield_string(ship, shield) (ship->shield_status[shield] == SHLD_UP ? "UP  " : "DOWN")

enum contact_value {NOT_SENSED, SENSED};

int check_pair(TAG *, TAG *);
void sense_huge_object(TAG *, TAG *);
TAG *nearest_huge_object(TAG *);
void new_sensor_turn(int);
void cleanup_contacts(int);
void update_contact(TAG *, TAG *, int);
CONTACT *add_contact(TAG *source, TAG *contact);
void new_contact(TAG *, CONTACT *);
void contact_lost(TAG *, CONTACT *, int);

void snsDisplayShortSensorInfo(TAG *senser, CONTACT *contact, dbref player, 
			       int caption);
static void build_contact_name(char *name, CONTACT *contact, int buflen);
void snsContactInfoString(TAG *senser, CONTACT *contact, char *buff);

static TAG* snsFindInsertPosPrev(TAG *tag, range_t new_range);
static TAG* snsFindInsertPosNext(TAG *tag, range_t new_range);

/* Used for detecting stale contacts */
static int sensor_turn_id;

/* Sensor routines */

/*
 * snsBuildDistanceTables:  Build and update a list of tags ordered by the
 *                          object range from the hub (0,0,0).
 */
void snsBuildDistanceTables(int space)
{
    TAG    *tag, *itag;
    range_t new_range;
    
#ifdef ENABLE_DEBUGGING
    /* Debugging feature - should pick up any weird features immediately.
       The range parameter should not be modified anywhere except here,
       so this one should work. The only possibility should be a newly
       added object failing the very first time. */
    snsSanityCheck(space);
#endif
    
    /* Cycle through the space list and generate the range table */
    for (tag=space_info[space].list; tag != NULL; tag = tag->next) {
	
	if (Removed(tag) || (tag->moveID != move_turn_id))
	    continue;
	
	new_range = HubRange(tag->pos);
	
	if ((new_range < tag->range) && (tag->range_prev != NULL)) {
	    
	    itag = snsFindInsertPosPrev(tag, new_range);
	    
	    if (itag != tag->range_prev) {

		tag->range_prev->range_next = tag->range_next;
		
		if (tag->range_next != NULL)
		    tag->range_next->range_prev = tag->range_prev;

		if (itag == NULL) {
		    tag->range_next = space_info[space].dist;
		    space_info[space].dist = tag;
		}
		else {
		    tag->range_next = itag->range_next;
		    itag->range_next = tag;
		}
		
		tag->range_prev = itag;
		
		if (tag->range_next != NULL)
		    tag->range_next->range_prev = tag;
		
	    }
	    
	} else if ((new_range > tag->range) && (tag->range_next != NULL)) {
	    
	    itag = snsFindInsertPosNext(tag, new_range);
	    
	    if (itag != tag) {
		
		if (tag->range_prev == NULL)
		    space_info[space].dist = tag->range_next;
		else
		    tag->range_prev->range_next = tag->range_next;
		
		tag->range_next->range_prev = tag->range_prev;
		
		tag->range_next = itag->range_next;
		tag->range_prev = itag;
		itag->range_next = tag;
		if (tag->range_next != NULL)
		    tag->range_next->range_prev = tag;
	    }
	}
	
	tag->moveID --;
	tag->range = new_range;
    }
    
#ifdef ENABLE_DEBUGGING
    /* Debugging feature - should pick up any weird features immediately.
       If an object is misplaced after the above has been run, something
       is seriously wrong. Either way, this should complain about it
       loudly in the space log. */
    snsSanityCheck(space);
#endif
    
    return;
}

/*
 * Finds the object to insert this item after in the range sorted list.
 */
TAG* snsFindInsertPosPrev(TAG *tag, range_t new_range)
{
    TAG *itag;
    
    for (itag = tag->range_prev; itag != NULL; itag = itag->range_prev) {
	
	if (itag->moveID == move_turn_id)
	    continue;
	
	if (itag->range <= new_range)
	    return itag;
    }

    return NULL;
}

/*
 * Finds the object to insert this item after in the range sorted list.
 */
TAG* snsFindInsertPosNext(TAG *tag, range_t new_range)
{
    TAG *itag;
    
    for (itag = tag; itag->range_next != NULL; itag = itag->range_next) {
	
	if (itag->range_next->moveID == move_turn_id)
	    continue;
	
	if (itag->range_next->range >= new_range)
	    return itag;
    }

    return itag;
}

/*
 * Adds the object to the linked list for the given space.
 */
void snsAddObject(TAG *object, int space)
{
    TAG    *tag;
    range_t new_range;
    
    /* Add object to the non-sorted list */
    object->next = space_info[space].list;
    space_info[space].list = object;
    
    /* Add object to the sorted list at the right place */
    tag = space_info[space].dist;

    if (tag == NULL) {

	/* If first item in list, just add it */
	space_info[space].dist = object;
	object->range_prev = NULL;
	object->range_next = NULL;

    } else {
    
	new_range = HubRange(object->pos);

	/* Search until last object in list, or insertion point found */
	while((tag->range < new_range) && (tag->range_next != NULL))
	    tag = tag->range_next;
	
	if (tag->range >= new_range) {
	    
	    /* Insert new object before tag */
	    object->range_prev = tag->range_prev;
	    object->range_next = tag;
	    
	    if (tag->range_prev != NULL)
		tag->range_prev->range_next = object;
	    else
		space_info[space].dist = object;
	    
	    tag->range_prev = object;
	
	} else {
	    
	    /* Insert new object after tag */
	    object->range_prev = tag;
	    object->range_next = tag->range_next;
	    if (object->range_next != NULL)
		object->range_next->range_prev = object;
	    
	    tag->range_next = object;
	}
    }
    
    if (Huge(object)) {
	object->huge_prev = NULL;
	object->huge_next = space_info[space].huge;
	
	if (object->huge_next != NULL)
	    object->huge_next->huge_prev = object;
	
	space_info[space].huge = object;
    } else {
	object->huge_prev = NULL;
	object->huge_next = NULL;
    }
    
    return;
}

void snsDelObject(TAG *object, int space)
{
    if (object->range_next != NULL)
	object->range_next->range_prev = object->range_prev;
    
    if (object->range_prev != NULL)
	object->range_prev->range_next = object->range_next;
    else
	space_info[space].dist = object->range_next;
    
    if (Huge(object)) {
	if (object->huge_next != NULL)
	    object->huge_next->huge_prev = object->huge_prev;
	
	if (object->huge_prev != NULL)
	    object->huge_prev->huge_next = object->huge_next;
	else
	    space_info[space].huge = object->huge_next;
    }
    
    return;
}

void snsCheckSensors(int space)
{
    TAG *entry, *ptr;
    
    /* Reset contacts state for new turn. */
    new_sensor_turn(space);
    
    for (entry=space_info[space].list; entry != NULL; entry=entry->next) {
	
	if (Removed(entry))
	    continue;
	
	if (CanSense(entry)) {
	    
	    /* Check objects forwards in the list */	
	    for (ptr=entry->range_next; ptr != NULL; ptr=ptr->range_next)
		if (check_pair(entry, ptr))
		    break;
	    
	    /* Check objects backwards in the list */	
	    for (ptr=entry->range_prev; ptr != NULL; ptr=ptr->range_prev) 
		if (check_pair(entry, ptr))
		    break;
	    
	    for (ptr=space_info[space].huge; ptr != NULL; ptr=ptr->huge_next)
		if (entry != ptr)
		    sense_huge_object(entry, ptr);
	}
    }
    
    /* Remove stale contacts */
    cleanup_contacts(space);
    
    return;
}

/* check_pair:  Ensure that a full sensor check is only done if sensee is
 *	        within sensor's sensor range.
 */
int check_pair(TAG *senser, TAG *sensee)
{
    if (Invisible(sensee)) {
	update_contact(senser, sensee, NOT_SENSED);
	return 0;
    }
    
    if (Omniscient(senser)) {
	update_contact(senser, sensee, SENSED);
	return 0;
    }
    
    /* Skip huge objects. They're processed separately. */
    if (Huge(sensee)) 
	return 0;
    
    /* See if sensee is even remotely in range */
    if ((RANGEA(sensee->range - senser->range) / 1.5) <= 
	senser->sensor_range) {
	
	snsSenseObject(senser, sensee);
	
	return 0;
    }
    
    /* Done looking in this direction */
    return 1;
}

/* 
 * sense_huge_object:  Use simple formula to see if huge object appears
 * 			on sensors.
 */
void sense_huge_object(TAG *senser, TAG *sensee)
{
    int result;
    range_t range;
    
    range = distance(&senser->pos, &sensee->pos);
    
    if ((range < (senser->sensor_range * sensee->size) ||
	 Omniscient(senser)) && !Invisible(sensee))
	result = SENSED;	
    else
	result = NOT_SENSED;
    
    update_contact(senser, sensee, result);
    
    return;
}

/*
 * snsSenseObject:  Use more complex formula to see if ship-sized object
 *		    appears on sensors.
 */
void snsSenseObject(TAG *senser, TAG *sensee)
{
    float effective_aspect_ratio; 	/* ease of sensing target */
    float prob;
    float planet_effect, weapons_effect;
    float modified_cloak_eff;
    TAG *hugeobj;
    int i;
    range_t hugerange, range;
    
    range = distance(&senser->pos, &sensee->pos);
    
    /*
     * If sensee is over three times the sensor range away from the senser,
     * don't bother doing any further calculations (to save time).
     */
    if (range > (3.0 * senser->sensor_range)) {
	update_contact(senser, sensee, NOT_SENSED);
	return;
    }
    
    if (Nearsighted(senser)) {
	if (range < senser->sensor_range)
	    update_contact(senser, sensee, SENSED);
	else
	    update_contact(senser, sensee, NOT_SENSED);
    }
    
    /*
     * Add cloak effect 
     */

    modified_cloak_eff = 1.0;
    if (Cloaked(sensee)) {
	modified_cloak_eff = sensee->cloak_effect * 10.0;
	
	if (Ship(sensee)) {
	    modified_cloak_eff += 10.0 *
		(100.0 - (float) sensee->shipdata->reactor_setting) /
		(float) sensee->shipdata->reactor_setting;
	}
    }
    
    effective_aspect_ratio = sensee->size / modified_cloak_eff;
    
    /* 
     * Planet effect makes uncloaked ships harder to see, but makes
     * cloaked ships easier to see.  Ignore if less than 5%, however.
     */
    
    hugeobj = nearest_huge_object(senser);
    
    if (hugeobj == NULL)
	planet_effect = 0;
    else {
	hugerange= distance(&sensee->pos, &hugeobj->pos);
	planet_effect = hugeobj->size * 0.80 /
	    (hugeobj->size + (hugerange * hugerange / 1000000.0));
    }
    
    if (planet_effect >= 0.05) {
	
	if (Cloaked(sensee)) 
	    effective_aspect_ratio *= (1.0 + (2 * planet_effect));
	else
	    effective_aspect_ratio *= (1.0 - planet_effect);
    }
    
    if (Ship(sensee)) {
	
	/*
	 * Weapons effect makes cloaked ships a lot easier to see, and
	 * makes seeing uncloaked ships somewhat easier.
	 */
	
	weapons_effect = (((float)(sensee->shipdata->num_torps_online)
			   *TORP_CHARGING_PENALTY) + 
			  ((float)(sensee->shipdata->num_guns_online) *
			   GUN_ONLINE_PENALTY));
	
	/* count armed torps */
	for (i=0; i < sensee->shipdata->number_of_torps; i++)
	       if (sensee->shipdata->torp[i].status == TORP_ARMED)
		   weapons_effect += TORP_ARMED_PENALTY;
    }
    else 
	weapons_effect = 0.0;
    
    if (Cloaked(sensee))
	effective_aspect_ratio += weapons_effect;
    else
	effective_aspect_ratio += weapons_effect * 0.25;
    
     if (Hazy(sensee))
	 effective_aspect_ratio *= 0.3;
     
     if (HasCataracts(senser))
	 effective_aspect_ratio *= 0.3;
     
     /* If sensee is locked on sensor, he's really easy to see. */
     if (sensee->locked_on != NULL)
	 if (sensee->locked_on->listref == senser)
	     effective_aspect_ratio += 10.0;
     
     if (sensee->pending_lock != NULL)
	 if (sensee->pending_lock->listref == senser)
	     effective_aspect_ratio += 10.0;
     
     /* 
      * If sensee has engaged a tractor beam on the senser, or vice-versa
      * he's also really easy to see.
      */
     
     if (Ship(senser) && senser->shipdata->tractor_target == sensee)
	 effective_aspect_ratio += 10.0;
     
     if (senser->tractor_source == sensee)
	 effective_aspect_ratio += 10.0;
     
     prob = 1.0 - range / (senser->sensor_range * effective_aspect_ratio);
     
     /* Warp speed adjustment */
     prob += ((sensee->speed - senser->speed) / 50.0);
     
     /* Bonus if senser has already made contact with sensee */
     if (snsFindContact(senser, sensee) != NULL)
	 prob += 0.25;
     
     if (senser->locked_on != NULL)
	 if (senser->locked_on->listref == sensee)
	     prob += 0.25;
     
     if (FRAND < prob)
	 update_contact(senser, sensee, SENSED);
     else
	 update_contact(senser, sensee, NOT_SENSED);
     
     return;
}

/*
 * nearest_huge_object:   Find the nearest 'huge' object.
 */
TAG *nearest_huge_object(TAG *senser)			
{
     TAG *nearobj = NULL;
     range_t nearrange, range;
     CONTACT *cptr;

     nearrange = MAX_RANGE;

     for (cptr=senser->contact_list;cptr != NULL; cptr=cptr->next) {
	  if (!Huge(cptr->listref))
	       continue;
	  range = distance(&senser->pos, &cptr->listref->pos);
	  if (range < nearrange || nearobj==NULL) {
	       nearobj = cptr->listref;
	       nearrange = range;
	  }
     }

     return nearobj;
}

/* Contact handling routines */

/*
 * new_sensor_turn:  Set all the 'updated' flags to FALSE on each contact
 *                   by incrementing the sensor_turn_id parameter.
 */
void new_sensor_turn(int space)
{
    sensor_turn_id++;
    
    return;
}

/* cleanup_contacts:  Remove contacts that haven't been updated this turn.
 *			Assume that they no longer exist. 
 */
void cleanup_contacts(int space)
{
    TAG *ptr;
    CONTACT *cptr, *cprev, *tmpptr;
    
    for (ptr=space_info[space].list; ptr != NULL; ptr=ptr->next) {
	
	cprev = NULL;
	cptr = ptr->contact_list;
	
	while (cptr != NULL) {
	    
	    if (cptr->updated != sensor_turn_id) {
		
		contact_lost(ptr, cptr, 0);
		if (cprev == NULL) 
		    ptr->contact_list = cptr->next;
		else
		    cprev->next = cptr->next;
		tmpptr = cptr;
		cptr = cptr->next;
		pse_free(tmpptr);
	    }
	    else {
		cprev = cptr;
		cptr = cptr->next;
	    }
	}
    }
    
    return;
    
}

/* snsRemoveContact:  Remove sensee.  */
void snsRemoveContact(TAG *senser, TAG *sensee, int silent)
{
    
    CONTACT *cprev = NULL;
    CONTACT *cptr = senser->contact_list;
    CONTACT *tmpptr;
    
    while (cptr != NULL) {
	
	if (cptr->listref == sensee) {
	    
	    contact_lost(senser, cptr, silent);
	    
	    if (cprev == NULL)
		senser->contact_list = cptr->next;
	    else
		cprev->next = cptr->next;
	    
	    tmpptr = cptr;
	    cptr = cptr->next;
	    pse_free(tmpptr);
	}
	else {	
	    cprev = cptr;
	    cptr = cptr->next;
	}
    }
    
    return;
}

void update_contact(TAG *senser, TAG *sensee, int action)
{
    CONTACT *contact;
    float prob, roll;
    int new_info_level, is_new_contact = 0;
    
    contact = snsFindContact(senser, sensee);
    
    if (action == SENSED) {
	
	if (contact == NULL) {
	    contact = add_contact(senser, sensee);
	    is_new_contact = 1;
	}
	
	if (contact->turns_of_contact < 65536)
	    contact->turns_of_contact++;
	
	contact->turns_since_last_contact = 0;
	contact->last_pos.x = contact->listref->pos.x;
	contact->last_pos.y = contact->listref->pos.y;
	contact->last_pos.z = contact->listref->pos.z;
	contact->updated = sensor_turn_id;
	
	if (contact->info_level == 5)
	    return;
	
	/* See how much information we can get. */
	prob = (float)(contact->turns_of_contact) / 20.0;
	roll = FRAND;
	
	if (Omniscient(senser) || Nearsighted(senser) || Huge(sensee))
	    roll = 0;
	
	if (roll < prob)
	    new_info_level = 5;
	else if (roll < (prob + .10))
	    new_info_level = 4;	
	else if (roll < (prob + .20))
	    new_info_level = 3;	
	else if (roll < (prob + .30))
	    new_info_level = 2;
	else
	    new_info_level = 1;
	
	/* Never show full info on cloakers */	
	if (Cloaked(contact->listref) && new_info_level > 3)
	    new_info_level = 3;
	
	if (is_new_contact)
	    new_contact(senser, contact);
	
	if (new_info_level > contact->info_level) {
	    contact->info_level = new_info_level;
	    if (!is_new_contact && Ship(senser))
		snsDisplaySensorInfo(senser, contact, -1, DSD_UPDATE);
	}
	
    } else {
	
	if (contact == NULL)
	    return;			
	
	contact->turns_since_last_contact++;
	
	if (!Cloaked(contact->listref) && !Invisible(contact->listref)
	    && !Huge(contact->listref) && 
	    contact->turns_since_last_contact < 3)
	    contact->updated = sensor_turn_id;
    }
    
    return;
}

/*
 * add_contact:  Add contact to contact list
 */
CONTACT *add_contact(TAG *source, TAG *contact)
{
    
    CONTACT *ptr;
    CONTACT *newcontact;
    
    newcontact = (CONTACT *) pse_malloc(sizeof(CONTACT));
    
    newcontact->listref = contact;
    newcontact->turns_of_contact = 0;
    newcontact->turns_since_last_contact = 0;
    newcontact->info_level = 0;
    newcontact->last_pos.x = 0;
    newcontact->last_pos.y = 0;
    newcontact->last_pos.z = 0;
    newcontact->watcher = 0;
    newcontact->inside_critical = FALSE;
    newcontact->next = NULL;
    
    if (source->contact_list==NULL) {
	
	source->contact_list = newcontact;
	source->contact_offset = 1;
	
    } else {
	for (ptr=source->contact_list; ptr->next != NULL; ptr=ptr->next)
	    ;
	ptr->next = newcontact;
	source->contact_offset++;
    }
    
    newcontact->contact_number = source->contact_offset;
    
    return(newcontact);
}

/*
 * snsFindContact:  Find contact in contact list
 */
CONTACT *snsFindContact(TAG *source, TAG *contact)
{
    CONTACT *ptr;
    
    for (ptr=source->contact_list; ptr != NULL; ptr=ptr->next) 
	if (ptr->listref==contact)
	    break;
    
    return(ptr);
}

CONTACT *snsFindContactByNumber(TAG *source, int num)
{
    CONTACT *ptr;
    
    for (ptr=source->contact_list; ptr != NULL; ptr=ptr->next) 
	if (ptr->contact_number == num)
	    break;
    
    return(ptr);
}

void new_contact(TAG *senser, CONTACT *contact)
{
    if (Ship(senser))
	snsDisplaySensorInfo(senser, contact, -1, DSD_NEW);
    
    if (EventDriven(senser)) 
	evTrigger(senser, EVENT_NEW_CONTACT, "c", contact);
    
#ifdef LOG_CONTACTS
    if (space_info[senser->space].flags & SPACE_LOGGED)
	log_space("NEW CONTACT!  %s (#%d) sees %s (#%d) at range " RANGEF ".",
		  senser->name, senser->data_object, contact->listref->name, 
		  contact->listref->data_object,
		  distance(&senser->pos, &contact->listref->pos));
#endif
}

/*
 * snsListContacts:  Spit contact list into buff, as either contact #'s 
 *		     (format = 0), or dbrefs (format = 1).  Show the entire
 *		     list (type=0), objects that can be beamed to (type=1),
 *		     and objects that can be beamed (type=2).
 */
void snsListContacts(char *buff, CONTACT *ptr, int type, int format)
{
    static char *buffstart = NULL;
    
    if (buffstart == NULL) {
	buffstart = buff;
	buff[0]='\0';
    }
    
    if ((buff - buffstart >= LARGE_BUF_SIZE - 10) || (ptr == NULL)) {
	if (buff[strlen(buff) - 1] == ' ')
	    buff[strlen(buff) - 1] = '\0';
	buffstart = NULL;
	return;
    }
    
    if ((type==0) || (type==1 && CanBeamTo(ptr->listref)) ||
	((type==2) && Beamable(ptr->listref))) {
	
	if (format)
	    sprintf(buff, "#%d ", ptr->listref->data_object);
	else
	    sprintf(buff, "%d ", ptr->contact_number);
	
    }
    
    snsListContacts(buff + strlen(buff), ptr->next, type, format);
    
    buffstart = NULL;
    return;
}

void snsDisplayContacts(TAG *object, dbref player, int show_ships,
			int show_other)
{
    CONTACT *contact;
    int found = 0;
    
    for (contact=object->contact_list; contact != NULL; contact=contact->next) {
	if (Ship(contact->listref) && !show_ships) continue;
	if (!Ship(contact->listref) && !show_other) continue;
	
	snsDisplaySensorInfo(object, contact, player, DSD_ROUTINE);
	found = 1;
	
    }
    
    if (!found)
	Notify(player, "No contacts on sensors.");
    
    return;
}

/*
 * Called by ship objects for new and updated contacts, and current contacts.
 * Called by non-ship objects current contacts only.
 */

void snsDisplaySensorInfo(TAG *senser, CONTACT *contact, dbref player, 
			  int control)
{
    char buff[SMALL_BUF_SIZE];
    XYZ xyz;
    SPH sph;
    TAG *target;
    
    /* Do some sanity checking. If it's not a ship and player is -1, we don't
       know where to emit messages to. */
    if ((player == -1) && !Ship(senser))
	return;
    
    /* calculate relative positions */
    xyz.x = contact->listref->pos.x - senser->pos.x;
    xyz.y = contact->listref->pos.y - senser->pos.y;
    xyz.z = contact->listref->pos.z - senser->pos.z;
    xyz_to_sph(xyz, &sph);
    
    target = contact->listref;
    
    /* display header message */
    switch (control) {
	
    case DSD_NEW:
	
	/* New contacts are only reported for ship objects */
	/* force the update to all tac consoles */
	player = -1;
	
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE,
		 "NEW CONTACT - designated number %d",
		 contact->contact_number);
	
	break;
	
    case DSD_UPDATE:
	
	/* Updated contacts are only reported for ship objects */
	/* force the update to all tac consoles */
	player = -1;
	
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE,
		 "Update:  Contact [%d] further identified as:",
		 contact->contact_number);
	break;
	
    case DSD_ROUTINE:
	
	/* Add a [*] if the target is locked onto us */
	if ((contact->listref->locked_on != NULL) &&
	    (contact->listref->locked_on->listref == senser)) 
	    sprintf(buff, "[*] Contact [%d]:", contact->contact_number);
	else
	    sprintf(buff, "Contact [%d]:", contact->contact_number);
	
	if (player == -1)
	    MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "%s",
		     buff);
	else
	    Notify(player, buff);
	
	break;
    }
    
    build_contact_name(buff, contact, SMALL_BUF_SIZE);
    if (player == -1)
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "%s", buff);
    else
	Notify(player, buff);
    
    sprintf(buff, "   Contact bearing %3.2f elevation %+2.2f range "
	    RANGEF_NP " %s",
	    sph.bearing, sph.elevation, sph.range * senser->range_factor,
	    senser->range_name);
    
    if (player == -1)
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "%s", buff);
    else
	Notify(player, buff);
    
    if (contact->info_level > 1)  {
	
	if (Ship(target) || target->speed > 0) {
	    /* construct the target's movement string */
	    if (player == -1)
		MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE,
			 "   Contact heading %3.2f%+2.2f at "
			 "warp %3.1f", target->heading.bearing,
			 target->heading.elevation,
			 target->speed);
	    else
		FNotify(player, "   Contact heading %3.2f%+2.2f at "
			"warp %3.1f", target->heading.bearing, 
			target->heading.elevation, 
			target->speed);
	}
	
	if (Ship(senser)) {
	    if (Ship(target))
		/* and give the shield facings */
		sprintf(buff, "   Our %s side is facing their %s side.",
			shdFullName(senser,
				    calcFacingShield(target->pos, senser)),
			shdFullName(target,
				    calcFacingShield(senser->pos, target)));
	    else 
		sprintf(buff, "   Our %s side is facing the contact.",
			shdFullName(senser,
				    calcFacingShield(target->pos,senser)));
	    
	    if (player == -1)
		MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "%s",
			 buff);
	    else
		Notify(player, buff);
	}
    }
    
    if (Cloaked(target)) {
	if (player == -1)
	    MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE,
		     "Contact is currently cloaked.");
	else
	    Notify(player, "Contact is currently cloaked.");
    }
    
    if (player == -1)
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "");
    else
	Notify(player, "");
    
    return;
}

void snsDisplayShortContacts(TAG *object, dbref player, int show_ships,
			     int show_other)
{
    CONTACT *contact;
    int found = 0;
    
    for (contact=object->contact_list; contact != NULL;
	 contact=contact->next) {
	if (Ship(contact->listref) && !show_ships) continue;
	if (!Ship(contact->listref) && !show_other) continue;
	
	if (Removed(contact->listref)) continue;
	
	if ((found % 22) == 0)
	    snsDisplayShortSensorInfo(object, contact, player, 1);
	else
	    snsDisplayShortSensorInfo(object, contact, player, 0);
	
	found ++;
    }
    
    if (!found)
	Notify(player, "No contacts on sensors.");
    
    return;
}

void snsDisplayShortSensorInfo(TAG *senser, CONTACT *contact, dbref player,
			       int caption)
{
     char line[SMALL_BUF_SIZE], name[SMALL_BUF_SIZE];
     char range[SMALL_BUF_SIZE], heading[SMALL_BUF_SIZE];
     const char *senser_shield;
     const char *target_shield;
     char locked;

     XYZ xyz;
     SPH sph;
     TAG *target;

     /* calculate relative positions */
     xyz.x = contact->listref->pos.x - senser->pos.x;
     xyz.y = contact->listref->pos.y - senser->pos.y;
     xyz.z = contact->listref->pos.z - senser->pos.z;
     xyz_to_sph(xyz, &sph);

     target = contact->listref;

     if ((contact->listref->locked_on != NULL) &&
	 (contact->listref->locked_on->listref == senser))
	  locked = '*';
     else
	  locked = ' ';

     build_contact_name(name, contact, SMALL_BUF_SIZE);

     if (Ship(senser))
	  senser_shield = shdCharName(senser,
				      calcFacingShield(target->pos,senser));
     else
	  senser_shield = " ";

     if (Ship(target))
	  target_shield = shdCharName(target,
				      calcFacingShield(senser->pos,target));
     else
	  target_shield = " ";

     if (caption) {

	  Notify(player, "##### Name                   Bearing       "
		 "Range   Heading       Speed Shields");
	  Notify(player, "----- ---------------------- ------------- "
		 "------- ------------- ----- -------");

     }

     navRangeString(senser, sph.range, range, 7);

     if (contact->info_level == 0)
	  strcpy(heading, "---.-- --.-- ---.- ");
     else
	  sprintf(heading, "%6.2f %+6.2f %+5.1f ", target->heading.bearing,
		  target->heading.elevation, target->speed);

     sprintf(line, "%c%4d %-22.22s %6.2f %+6.2f %7.7s %19.19s  %2.2s %2.2s",
	     locked, contact->contact_number, name,
	     sph.bearing, sph.elevation, range,
	     heading, senser_shield, target_shield);

     Notify(player, line);

     return;

}

void build_contact_name(char *name, CONTACT *contact, int buffsize)
{

     char *fargs[1];
     SHIP *ship;
     TAG *target;

     target = contact->listref;
     ship = target->shipdata;

     fargs[0]=AllocBuffer("evDisplay");
     sprintf(fargs[0], "%d", contact->info_level);
     strncpy(name, getEvalAttr(target->data_object, INFO_CONTACT_STRING, 
			      fargs, 1), buffsize-1);
     name[buffsize-1] = '\0';
     
     FreeBuffer(fargs[0]);

     if (strlen(name)==0)
	  strcpy(name, target->name);
     else
          return;

     if (!Ship(target))
	  return;

     switch (contact->info_level) {

     case 5:
	  sprintf(name, "%s -- %s %s-class %s", target->name,
		  ship->owner_name, ship->class, ship->type);

	  break;

     case 4:
	  sprintf(name, "%s %s-class %s", ship->owner_name,
		  ship->class, ship->type);
	  break;
     case 3:
	  sprintf(name, "%s %s", ship->owner_name, 
		  ship->type);
	  break;
     default:
	  sprintf(name, "%s", ship->type);
	  break;
     }
}

void snsContactString(TAG *senser, CONTACT *contact, char *buff)
{
     char name[SMALL_BUF_SIZE], head[SMALL_BUF_SIZE];
     const char *senser_shield;
     const char *target_shield;

     TAG *target;
     XYZ xyz;
     SPH sph;

     target = contact->listref;

     /* calculate relative positions */
     xyz.x = target->pos.x - senser->pos.x;
     xyz.y = target->pos.y - senser->pos.y;
     xyz.z = target->pos.z - senser->pos.z;
     xyz_to_sph(xyz, &sph);


     /* Work out the name of the contact based on visibility etc */
     build_contact_name(name, contact, SMALL_BUF_SIZE);

     if (contact->info_level == 0)
	  strcpy(head, "? +? ?");
     else
	  sprintf(head, "%.2f %+.2f %.1f", target->heading.bearing,
		  target->heading.elevation, target->speed);

     senser_shield = shdCharName(senser,calcFacingShield(target->pos,senser));
     target_shield = shdCharName(target,calcFacingShield(senser->pos,target));

     if (!Ship(target) || (contact->info_level == 0))
	  target_shield = "-";

     sprintf(buff, "SIZE: %.2f VISBILITY: %d BEARING: %.2f %+.2f %f "
	     "HEADING: %s FACING: %2.2s %2.2s NAME: %s",
	     target->size, contact->info_level,
	     sph.bearing, sph.elevation, sph.range * senser->range_factor,
	     head, senser_shield, target_shield, name);
}

void snsScan(TAG *object, dbref player, int num)
{

     XYZ xyz;
     SPH sph;
     SHIP *scannee;
     CONTACT *source;
     CONTACT *target;
     char *fargs[1];
     const char *rb;

     if (object->shipdata->damage[SYSTEM_SCANNERS].status > '6') {
	  Notify(player, "Scanners are damaged and inoperable.");
	  return;
     }

     target = snsFindContactByNumber(object, num);

     if (target==NULL) {
	  Notify(player, "Invalid target.");
	  return;
     }

     if (distance(&object->pos, &target->listref->pos) > 
	 object->shipdata->scanner_range) {
	  Notify(player, "That object is out of scanner range.");
	  return;
     }

     /* Raise our info level on the target object. */

     if (Cloaked(target->listref))
	  target->info_level = Max(target->info_level, 4);
     else
	  target->info_level = 5;

     FNotify(player, "Scanning [%d]: %s", target->contact_number,
	     target->listref->name);

     xyz.x = target->listref->pos.x - object->pos.x;
     xyz.y = target->listref->pos.y - object->pos.y;
     xyz.z = target->listref->pos.z - object->pos.z;
     xyz_to_sph(xyz, &sph);

     /*
      * Scanning info strings always relies on the value from the evaluation.
      * If you need it to emulate the old behaviour, just return the desc from
      * the eval.
      */
     fargs[0]=AllocBuffer("snsScan");
     sprintf(fargs[0], "%d", target->contact_number);
     rb = getEvalAttr(object->data_object, INFO_SCAN_STRING, fargs, 1);
     if (rb && *rb)
	 Notify(player, rb);
     
     FreeBuffer(fargs[0]);
     
     /* We're done for non-ship objects */
     if (!Ship(target->listref))
	  return;
     
     scannee = target->listref->shipdata;

     FNotify(player, "Target bearing %3.2f%+2.2f at range %.f %s",
	     sph.bearing, sph.elevation, sph.range * object->range_factor,
	     object->range_name);

     FNotify(player, "       heading %3.2f%+2.2f at warp %1.0f",
	     target->listref->heading.bearing,
	     target->listref->heading.elevation,
	     target->listref->speed);	

     /* weapons status */

     Notify(player, "Weapons:");
     FNotify(player, "    %d %ss online -- energy allocated %d",
	     scannee->num_guns_online, scannee->gun_string,
	     scannee->talloc_guns);
     FNotify(player, "    %d %ss online -- energy allocated %d",
	     scannee->num_torps_online, scannee->torp_string,
	     scannee->talloc_torps);
     Notify(player, " ");

     if (Cloaked(target->listref))
	  Notify(player, "Target's cloaking device is engaged.");
     else {
	  Notify(player, "Shields:");
	  switch(scannee->shield_number) {
	  case 4:
	       FNotify(player, "%s: %d:%s  %s: %d:%s  %s: %d:%s  %s: %d:%s",
		       shdShortName(target->listref, 0),
		       scannee->shield_level[0], Shield_string(scannee, 0),
		       shdShortName(target->listref, 1),
		       scannee->shield_level[1], Shield_string(scannee, 1),
		       shdShortName(target->listref, 2),
		       scannee->shield_level[2], Shield_string(scannee, 2),
		       shdShortName(target->listref, 3),
		       scannee->shield_level[3], Shield_string(scannee, 3));
	       break;
	  case 6:
	       FNotify(player, "%s: %d:%s  %s: %d:%s  %s: %d:%s  %s: %d:%s  "
		       "%s: %d:%s  %s: %d:%s",
		       shdShortName(target->listref, 0),
		       scannee->shield_level[0], Shield_string(scannee, 0),
		       shdShortName(target->listref, 1),
		       scannee->shield_level[1], Shield_string(scannee, 1),
		       shdShortName(target->listref, 2),
		       scannee->shield_level[2], Shield_string(scannee, 2),
		       shdShortName(target->listref, 3),
		       scannee->shield_level[3], Shield_string(scannee, 3),
		       shdShortName(target->listref, 4),
		       scannee->shield_level[4], Shield_string(scannee, 4),
		       shdShortName(target->listref, 5),
		       scannee->shield_level[5], Shield_string(scannee, 5));
	       break;
	  default:
	       FNotify(player, "Illegal number of shields: %d",
		       scannee->shield_number);
	       break;
	  }
     }
     
     FNotify(player, "We are currently facing their %s shield.",
	     shdFullName(target->listref, calcFacingShield(object->pos,
							   target->listref)));

     FNotify(player, "They are currently facing our %s shield.",
	     shdFullName(object, calcFacingShield(target->listref->pos,
						  object)));

     Notify(player, " ");

     if (scannee->door_status != DOORS_NONE) {
	  FNotify(player, "The target's cargo bay doors are %s.",
		  scannee->door_status == DOORS_CLOSED ? "closed" : "open");

	  Notify(player, " ");
     }

     FNotify(player, "Hull integrity:  %2.0f%%\tpower to warp: %d",
	     (float)scannee->current_integrity / (float)scannee->hull_integrity
	     * 100.0, scannee->alloc_nav);

     source = snsFindContact(target->listref, object);

     if (source != NULL)
	  MFNotify(scannee, -1, CONS_TAC, CONS_ACTIVE,
		   "We are being scanned by contact [%d]: %s.",
		   source->contact_number, object->name);
     else
	  MFNotify(scannee, -1, CONS_TAC, CONS_ACTIVE,
		   "We are being scanned by an unknown source.");

     return;

}

void contact_lost(TAG *senser, CONTACT *contact, int silent)
{
    char buff[SMALL_BUF_SIZE];
    XYZ last_position;
    SPH last_relative;
    
    if (Ship(senser) && !silent) {
	if (senser->shipdata->hull_integrity < 0)
	    return;
	
	MFNotify(senser->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		 "CONTACT [%d] LOST:",contact->contact_number);
	build_contact_name(buff, contact, SMALL_BUF_SIZE);
	MFNotify(senser->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		 "%s", buff);
	
	last_position.x = contact->last_pos.x - senser->pos.x;
	last_position.y = contact->last_pos.y - senser->pos.y;
	last_position.z = contact->last_pos.z - senser->pos.z;
	
	xyz_to_sph(last_position, &last_relative);
	
	MFNotify(senser->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		 "Contact last spotted bearing: "
		 "%3.0f%+2.0f, range %.f %s", last_relative.bearing,
		 last_relative.elevation,
		 last_relative.range * senser->range_factor,
		 senser->range_name);
	
	if (contact->info_level > 1) 
	    MFNotify(senser->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		     "                     heading: %3.0f"
		     "%+2.0f at warp %3.1f", 
		     contact->listref->heading.bearing,
		     contact->listref->heading.elevation, 
		     contact->listref->speed);
    }
    
    if (senser->locked_on == contact)
	tacBreakLock(senser);
    
    if (Ship(senser))
	if (senser->shipdata->tractor_target == contact->listref)
	    tacBreakTractor(senser);
    
    /*
     * If the lock is broken before it is achieved, we can simulate a
     * message to send to the tacofficer, by granting the lock then breaking
     * it.
     */
    
    if (senser->pending_lock == contact) {
	senser->locked_on = contact;
	tacBreakLock(senser);
	senser->pending_lock = NULL;
    }
    
    if (EventDriven(senser) && !silent)  {
	
	if (contact->inside_critical)
	    evTrigger(senser, EVENT_OUTSIDE_CRITICAL, "c", contact);
	
	evTrigger(senser, EVENT_CONTACT_LOST, "c", contact);
    }
    
#ifdef LOG_CONTACTS
    if ((space_info[senser->space].flags & SPACE_LOGGED) && !silent)
	log_space("CONTACT LOST.  %s (#%d) no longer sees %s (#%d), "
		  "range " RANGEF ".", senser->name, senser->data_object, 
		  contact->listref->name, contact->listref->data_object,
		  distance(&senser->pos, &contact->listref->pos));
#endif
    
    return;
}

/*
 * Do some diagnostics on the sensor related information.
 */
void snsSanityCheck(int space)
{
    CONTACT *ct;
    TAG     *tag, *dtag;
    int     tc, dc;
    
    for (tag = space_info[space].list, tc=0; tag != NULL; tag=tag->next)
	tc++;
    
    for (tag = space_info[space].dist, dc=0; tag != NULL; tag=tag->range_next)
	dc ++;
    
    if (tc != dc)
	log_space("Insanity %d: tc(%d) != dc(%d)", space, tc, dc);
    
    for (tag = space_info[space].list; tag != NULL; tag=tag->next) {
	for (dtag = space_info[space].dist; (dtag != tag) && (dtag != NULL);
	     dtag = dtag->range_next)
	    ;
	
	if (!dtag)
	    log_space("Insanity %d: Object %d not on distance list.",
		      space, tag->data_object);
    }
    
    for (tag = space_info[space].dist; tag != NULL; tag=tag->range_next) {
	for (dtag = space_info[space].list; (dtag != tag) && (dtag != NULL);
	     dtag = dtag->next)
	    ;
	
	if (!dtag)
	    log_space("Insanity %d: Object %d not on space object list.",
		      space, tag->data_object);
    }
    
    for (tag = space_info[space].list; tag != NULL; tag=tag->next)
	for (dtag = tag->next; dtag != NULL; dtag=dtag->next)
	    if (dtag->data_object == tag->data_object)
		log_space("Insanity %d: Object %d in space list twice.",
			  space, tag->data_object);
    
    for (tag = space_info[space].dist; tag != NULL; tag=tag->range_next)
	for (dtag = tag->range_next; dtag != NULL; dtag=dtag->range_next)
	    if (dtag->data_object == tag->data_object)
		log_space("Insanity %d: Object %d in distance list twice.",
			  space, tag->data_object);
    
    for (tag = space_info[space].dist; tag != NULL; tag=tag->range_next)
	if ((tag->range_next) && (tag->range_next->range < tag->range)) {
	    log_space("Insanity %d (%d): Objects %d (%d:%d) / %d (%d:%d) misplaced "
		      "in range list.", space, move_turn_id,
		      tag->data_object,
		      tag->moveID,
		      tag->range,
		      tag->range_next->data_object,
		      tag->range_next->moveID,
		      tag->range_next->range);
	    /* These should automagically fix the error next sensor recalc */
	    /*	    tag->moveID = move_turn_id + 1;
		    tag->range_next->moveID = move_turn_id + 1; */
	}
    
    for (tag = space_info[space].list; tag != NULL; tag=tag->next) {
	for (ct = tag->contact_list; ct != NULL; ct=ct->next) {
	    if (ct->info_level == 0)
		log_space("Insanity %d: Object %d losing contact with %d",
			  space, tag->data_object, ct->listref->data_object);
	}
    }
}
