/* v0.8
 *
 * object.c:  Space object code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <stdlib.h>
#include "hashtab.h"
#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "damage.h"
#include "smisc.h"
#include "events.h"
#include "sensors.h"

TAG *space_list[NUM_SPACES];
TAG *space_tail[NUM_SPACES];

typedef struct {

        char    *name;
        int     num;

} SFLAGENT;
  
SFLAGENT objflaglist[] = {
{"SHIP",		TYPE_SHIP},
{"PLANET",		TYPE_PLANET},
{"CAN_MOVE",		CAN_MOVE},
{"MANUAL_POS",		MANUAL_POS},
{"MANUAL_DIR",		MANUAL_DIR},
{"EVENT_DRIVEN",	EVENT_DRIVEN},
{"CAN_SENSE",		CAN_SENSE},
{"CAN_BEAM_TO",		CAN_BEAM_TO},
{"BEAMABLE",		BEAMABLE},
{"ATTACKABLE",		ATTACKABLE},
{"HUGE",		HUGE_OBJ},
{"OMNISCIENT",		OMNISCIENT},
{"NEARSIGHTED",		NEARSIGHTED},
{"INVISIBLE",		INVISIBLE},
{"HAZY",		HAZY},
{"CATARACTS",		CATARACTS},
{"CLOAKED",		CLOAKED},
{"CAN_BEAM_FROM",	CAN_BEAM_FROM},
{"REMOVED",			REMOVED},
{"",			0}};

SFLAGENT shipflaglist[] = {
{"INVINCIBLE",		INVINCIBLE},
{"SPEEDY",		SPEEDY},
{"POKEY",		POKEY},
{"NAKED",		NAKED},
{"DISABLED",		DISABLED},
{"PACIFIST",		PACIFIST},
{"CAN_LAND",		CAN_LAND},
{"",			0}};

HNODE *shipflag_htab;
HNODE *objflag_htab;

SHIP *load_ship(dbref);
void load_flags(int *, char *);
char *list_flags(int, SFLAGENT *);
int set_flag(int *, int, HNODE *, char *);
int is_set(int, HNODE *, char *);

void objInitSpace(void)
{
	int i;

	for (i=0; i < NUM_SPACES; i++) 
		space_list[i] = NULL;

	shipflag_htab = NULL;
	objflag_htab = NULL;

	for (i=0; *(objflaglist[i].name) != '\0'; i++)
		hashInsert(objflaglist[i].name, (int *)(objflaglist + i), 
		  &objflag_htab);

	for (i=0; *(shipflaglist[i].name) != '\0'; i++)
		hashInsert(shipflaglist[i].name, (int *)(shipflaglist + i),
		  &shipflag_htab);

	return;
}

void objAddObject(dbref data)
{
	int space;
	TAG *new_object, *ptr;
	int start_dbref;

	if (objFindObject(data)) {
		log_space("AddObject Failure:  #%d:  Object already in space.",
		  data);
		return;
	}

	if (!(new_object=(TAG *)malloc(sizeof(TAG)))) {
		log_space("AddObject Failure:  #%d:  Can't allocate new space 
		  object.", data);
		return;
	}

	/* Initialize the new object */

	new_object->data_object = data;
	space = new_object->space = atoi(getAttrByName(data, STATUS_SPACE));

	strncpy(new_object->name, getAttrByNumber(data, A_NAME), NAME_LEN-1);
	new_object->name[NAME_LEN-1] = '\0';
	
	if (space < 0 || space >= NUM_SPACES) {
		log_space("AddObject Failure:  #%d:  Bad space number.  Valid 
		  range is 0 to %d.", 
		  data, NUM_SPACES - 1);
		free(new_object);
		return;
	}

	load_flags(&(new_object->tagflags), getAttrByName(data, STATUS_FLAGS));

	start_dbref = parseDbref(getAttrByName(data, STATUS_DOCKED_AT));

	ptr = NULL;
	if (start_dbref >= 0) 
		ptr = objFindObject(start_dbref);
	
	if (ptr != NULL) {
		new_object->pos.x = ptr->pos.x;
		new_object->pos.y = ptr->pos.y;
		new_object->pos.z = ptr->pos.z;
	}
	else
		fillXYZCoords(data, STATUS_STARTPOS, &(new_object->pos));

	new_object->heading.bearing = 0.0;
	new_object->heading.elevation = 0.0;
	new_object->heading.range = 1;

	new_object->heading_adj.bearing = 0;
	new_object->heading_adj.elevation = 0;
	new_object->heading_adj.range = 0;

	new_object->speed = 0.0;
	new_object->size = atof(getAttrByName(data, CONFIG_SIZE));
	new_object->tractor_source = NULL;

	if (new_object->size <= 0) {
		log_space("AddObject Failure:  #%d:  Object size must be "
		  "greater than zero.", data);
		free(new_object);
		return;
	}

	new_object->sensor_range = atoi(getAttrByName(data, 
	  CONFIG_SENSOR_RANGE));
	new_object->transporter_range = atoi(getAttrByName(data, 
	  CONFIG_XPORT_RANGE));
	new_object->critical_range = atoi(getAttrByName(data, 
	  CONFIG_CRITICAL_RANGE));
	new_object->cloak_effect = atof(getAttrByName(data, 
	  CONFIG_CLOAK_EFFECT));

	new_object->pending_lock = NULL;
	new_object->locked_on = NULL;
	new_object->contact_list = NULL;

	if (Ship(new_object)) {
		if ((new_object->shipdata = load_ship(data)) == NULL) {
			free(new_object);
			return;
		}

		damInitSystemDamage(new_object);
	}
	else
		new_object->shipdata = NULL;

	/* Make sure sensor range is nonzero */
	if (new_object->sensor_range == 0.0) {
		log_space("AddObject Failure:  #%d:  Bad sensor range", data);
		return;
	}

	new_object->next = NULL;

	/* Link in new object */

	if (space_list[space]==NULL)
		space_list[space] = new_object;
	else {
		for (ptr=space_list[space]; ptr->next != NULL; ptr=ptr->next)
			;
		ptr->next = new_object;
	}

	if (space == 0)
		log_space("Added %s (#%d) to real space.", 
		  new_object->name,
		  data); 

	evStartup(new_object);

	return;	

}

void objRemoveObject(TAG *object)
{

	evShutdown(object);

	object->tagflags |= REMOVED;

	return;

}	

void objFreeMarkedObjects(int space)
{
	TAG *object, *prev;
	TAG *ptr;

	object = space_list[space];
	prev = NULL;
	
	while(object != NULL) {
		/* If we are removing this object */
		if (Removed(object)) {

			/* Remove ships from contact lists of other ships in
			   its space */
			for (ptr = space_list[space]; ptr != NULL; ptr=ptr->next)
				if (ptr != object)
					snsRemoveContact(ptr, object, 0);

			/* Remove ship's own contacts */
			while (object->contact_list != NULL)
				snsRemoveContact(object, object->contact_list->listref, 1);

			/* remove object from space list */
			if (prev)
				prev->next = object->next;
			else
				space_list[space] = object->next;

			if (Ship(object)) {
				damWriteDamage(object->data_object, object->shipdata);

				free(object->shipdata);
			}

			if (space == 0)
				log_space("Removed %s (#%d) from real space.", 
					object->name, object->data_object); 

			ptr = object->next;

			free(object);

			/* Rebuild distance tables for this space */
			snsFreeDistanceTables(space);
			snsBuildDistanceTables(space);	

			object = ptr;
		}
		else {
			prev = object;
			object = object->next;
		}
	}

	return;
}

SHIP *load_ship(dbref data)
{

	char buff[SMALL_BUF_SIZE];
	SHIP *ship;
	int t1, t2, i;

	if ((ship=(SHIP *)malloc(sizeof(SHIP)))==NULL) {
		log_space("#%d:  Can't allocate ship.", data);
		return NULL;
	}

	ship->tactical = getDbrefByName(data, CONFIG_DBREF_TACTICAL);
	ship->shields = getDbrefByName(data, CONFIG_DBREF_SHIELDS);
	ship->nav = getDbrefByName(data, CONFIG_DBREF_NAV);
	ship->eng = getDbrefByName(data, CONFIG_DBREF_ENG);
	ship->trans = getDbrefByName(data, CONFIG_DBREF_XPORT);
	ship->comm = getDbrefByName(data, CONFIG_DBREF_COMM);
	ship->damcon = getDbrefByName(data, CONFIG_DBREF_DAMCON);

	if (!ValidObject(ship->tactical) || !ValidObject(ship->nav) || 
	!ValidObject(ship->shields) || !ValidObject(ship->eng) || 
	!ValidObject(ship->trans) || !ValidObject(ship->comm) || 
	!ValidObject(ship->damcon)) {
		log_space("#%d:  Invalid console dbref.", data);
		free(ship);
		return NULL;
	}
	
	 /* initialize damage registers */
	strncpy(buff, getAttrByName(data, STATUS_SYSTEM_DAMAGE), SMALL_BUF_SIZE);
	buff[SMALL_BUF_SIZE - 1] = '\0';
	
	for (i = 0; i < NUM_SYSTEMS; i++) {
		if (buff[i] == 'X')
			buff[i] = 'x';

		if (((buff[i] < '0') || (buff[i] > '9'))
			       && (buff[i] != 'x'))
			ship->damage[i].status = '0';
		else
			ship->damage[i].status = buff[i];

		ship->damage[i].teams = 0;
		ship->damage[i].maxteams = 2;
		ship->damage[i].time_to_fix = 0;

  	}

	ship->damage[SYSTEM_HULL].maxteams = 4;

	load_flags(&(ship->shipflags), getAttrByName(data, 
	  STATUS_SHIPFLAGS));

	strncpy(ship->class, getAttrByName(data, CONFIG_CLASS), 
	  CLASS_LEN-1);
  	strncpy(ship->type, getAttrByName(data, CONFIG_TYPE), 
	  TYPE_LEN-1);
	strncpy(ship->owner_name, getAttrByName(data, CONFIG_OWNER_NAME),
	  OWNER_LEN-1);

	ship->class[CLASS_LEN-1] = '\0';
	ship->type[TYPE_LEN-1] = '\0';
	ship->owner_name[OWNER_LEN-1] = '\0';

	ship->owner = atoi(getAttrByName(data, CONFIG_OWNER_NUM));

	/* Docking bay doors */

	ship->door_status = atoi(getAttrByName(data, CONFIG_DOOR_STATUS))
	  ? DOORS_CLOSED : DOORS_NONE; 

	if (ship->door_status != DOORS_NONE) {
		ship->door_size = atof(getAttrByName(data, CONFIG_DOOR_SIZE));
		ship->door_side = atoi(getAttrByName(data, CONFIG_DOOR_SIDE));	
		if (ship->door_side < 0 || ship->door_side > 3) {
			log_space("Door side must be between 0 and 3.");
			free(ship);
			return NULL;
		}
	}

	/* Engine stuff */

	ship->hull_integrity = atoi(getAttrByName(data, CONFIG_HULL));
	ship->reactor_output_max = atoi(getAttrByName(data, 
	  CONFIG_REACTOR_OUTPUT));

	ship->max_overload_points = atoi(getAttrByName(data, 
	  CONFIG_REACTOR_OVER_POINTS));
	ship->reactor_setting_limit = atoi(getAttrByName(data, 
	  CONFIG_REACTOR_LEVEL_STRESS));
	ship->reactor_stress_level = atoi(getAttrByName(data, 
	  CONFIG_REACTOR_LEVEL_NORMAL));
	ship->reactor_overload_penalty = atoi(getAttrByName(data, 
	  CONFIG_REACTOR_OVER_PENALTY));
	
	/* battery stuff */
	ship->battery_capacity = atoi(getAttrByName(data, 
	  CONFIG_BATTS_CAPACITY));
	ship->battery_discharge_max = atoi(getAttrByName(data, 
	  CONFIG_BATTS_DISCHARGE));

	/* general ship characteristics */
	ship->warp_factor = atof(getAttrByName(data, CONFIG_WARP_FACTOR));
	ship->warp_max = atof(getAttrByName(data, CONFIG_WARP_MAX));

	/* cloak */
	ship->cloak_status = atoi(getAttrByName(data, CONFIG_CLOAK_STATUS)) 
	  ? CLOAK_OFF : CLOAK_NONE;
	ship->time_to_cloak = 0;
	ship->cloak_cost = atoi(getAttrByName(data, CONFIG_CLOAK_COST));

	/* sensors */
	ship->scanner_range = atoi(getAttrByName(data, CONFIG_SCANNER_RANGE));
 
	/* comm */
	ship->transmitter_range = atof(getAttrByName(data, CONFIG_COMM_RANGE));
	ship->receiver_sensitivity = atof(getAttrByName(data, 
	  CONFIG_COMM_SENSITIVITY));

	/* shield stuff */
	ship->shield_factor = atof(getAttrByName(data, CONFIG_SHIELD_FACTOR));

	for (i = 0; i < 4; i++) {
		ship->shield_max[i] = atoi(getAttrByName(data, 
		  CONFIG_SHIELD_POWER));
		ship->shield_max_charge[i] = atoi(getAttrByName(data, 
		  CONFIG_SHIELD_CHARGE_RATE));
	}

	/* gun stuff */
	strcpy(ship->gun_string, getAttrByName(data, CONFIG_GUN_NAME));
	ship->number_of_guns = atoi(getAttrByName(data, CONFIG_GUN_QTY));
	if (ship->number_of_guns > MAX_GUNS)
		ship->number_of_guns = MAX_GUNS;

	for (i = 0; i < ship->number_of_guns; i++) {
		ship->gun[i].status = GUN_OFFLINE;
		ship->gun[i].power = atoi(getAttrByName(data, 
		  CONFIG_GUN_POWER));
		ship->gun[i].range = atoi(getAttrByName(data, 
		  CONFIG_GUN_RANGE));
		ship->gun[i].range_percent = atof(getAttrByName(data, 
		  CONFIG_GUN_DELIVERY));
		ship->gun[i].charge_per_turn = atoi(getAttrByName(data, 
		  CONFIG_GUN_CHARGE_RATE));
  	}

	/* torp stuff */
	strcpy(ship->torp_string, getAttrByName(data, CONFIG_TORP_NAME));
	ship->number_of_torps = atoi(getAttrByName(data, CONFIG_TORP_QTY));
	if (ship->number_of_torps > MAX_TORPS)
		ship->number_of_torps = 0;
	strcpy(buff, getAttrByName(data, CONFIG_TORP_AMMO));
	if (strlen(buff) > 0)
		ship->torp_ammo = atoi(buff);
	else
		ship->torp_ammo = -1;

	for (i = 0; i < ship->number_of_torps; i++) {
		ship->torp[i].power = atoi(getAttrByName(data, 
		  CONFIG_TORP_POWER));
		ship->torp[i].range = atoi(getAttrByName(data, 
		  CONFIG_TORP_RANGE));
		ship->torp[i].charge_per_turn = atoi(getAttrByName(data, 
		  CONFIG_TORP_CHARGE_RATE));
		ship->torp[i].base_accuracy = atof(getAttrByName(data, 
		  CONFIG_TORP_ACCURACY));
	}

	ship->num_torp_loaders = atoi(getAttrByName(data,
	  CONFIG_TORP_LOADERS));

	ship->free_torp_loaders = ship->num_torp_loaders;
	ship->tractor_status = atoi(getAttrByName(data, CONFIG_TRACTOR_STATUS))
	  ? TRACTOR_OFF : TRACTOR_NONE; 

	if (ship->tractor_status != TRACTOR_NONE)
		ship->tractor_effect = atof(getAttrByName(data, 
		  CONFIG_TRACTOR_EFFECT));

	/* initialize damage control team status */
	ship->damcon_teams = atoi(getAttrByName(data, CONFIG_DAMCON_TEAMS));
	if (ship->damcon_teams > MAX_TEAMS)
		ship->damcon_teams = MAX_TEAMS;

	sscanf(getAttrByName(data, STATUS_HULL_DAMAGE), "%d %d", 
	  &t1, &t2);
	ship->current_integrity = ship->hull_integrity - t1;
	ship->permanent_hull_damage = t2;

	if (ship->current_integrity < 0)
		ship->shipflags |= DISABLED;

	ship->reactor_setting = 5;
	ship->reactor_drain = 0;
	ship->overload_points = 0;
	ship->eng_warnings = 1; /* set warnings on */
	ship->eng_safeties = 1; /* set automatic overrides on */
	ship->battery_discharge = 0;
	ship->battery_level = ship->battery_capacity;

	/* cloak */
	if (ship->cloak_status == CLOAK_NONE)
		ship->damage[SYSTEM_CLOAK].status = 'x';

	/* tractor */
	if (ship->tractor_status == TRACTOR_NONE)
		ship->damage[SYSTEM_TRACTOR].status = 'x';

	/* default allocations */
	ship->alloc_nav = 0;
	ship->alloc_batt = 0;
	ship->alloc_tac = 0;
	ship->alloc_nav = 0;
	ship->alloc_shields = 0;
	ship->talloc_guns = 0;
	ship->talloc_torps = 0;
	ship->talloc_tractor = 0;
	ship->tractor_target = NULL;
	ship->tractor_power = 0;
	ship->num_guns_online = 0;
	ship->num_torps_online = 0;

	ship->warp_speed = 0.0;

	/* shield stuff */
	for (i = 0; i < 4; i++) {
		ship->shield_level[i] = 0;
		ship->shield_alloc[i] = 0;
		ship->shield_status[i] = SHLD_READY;
		ship->shield_action[i] = SHLD_STABLE;
	}

	/* gun stuff */
	for (i = 0; i < ship->number_of_guns; i++) {
		ship->gun[i].status = GUN_OFFLINE;
		ship->gun[i].charge = 0;
	}

	for (i = ship->number_of_guns; i < MAX_GUNS; i++)
		ship->damage[SYSTEM_GUN_0 + i].status = 'x';

	/* torp stuff */
	ship->auto_online = FALSE;
	ship->auto_reload = TRUE;

	for (i = 0; i < ship->number_of_torps; i++) {
		ship->torp[i].status = TORP_LOADED;
		ship->torp[i].charge = 0;
		ship->torp[i].turns_charged = 0;
	}

	for (i = ship->number_of_torps; i < MAX_TORPS; i++)
		ship->damage[SYSTEM_TORP_0 + i].status = 'x';

	/* initialize comm configuration */
	for (i = 0; i < MAX_CHANNELS; i++) {
		ship->channel[i].frequency = 0;
		ship->channel[i].send_status = CS_COMM_ONLY;
		ship->channel[i].receive_status = CR_COMM_ONLY;
		strcpy(ship->channel[i].label, "");
	}

	/* initialize damage control team status */
	for (i = 0; i < ship->damcon_teams; i++)
		ship->team[i] = -1;

	if (ship->max_overload_points == 0)
		log_space("#%d:  Bad max_overload_points", data);
	else if (ship->warp_factor == 0.0)
		log_space("#%d:  Bad warp_factor", data);
	else if (ship->hull_integrity == 0)
		log_space("#%d:  Bad hull_integrity", data);
	else
		return ship;  /* everything checks out */

	/* if it gets to here there is a problem with the ship object */
	free(ship);
	return NULL;

}

void objDisplaySpecs(TAG *object, dbref player, int godmode)
{
	SHIP *ship = object->shipdata;

	if (Ship(object))
		FNotify(player, "%s -- %s %s-class %s", object->name,
		  ship->owner_name, ship->class, ship->type);
	else
		FNotify(player, "%s", object->name);

	if (godmode) {
		FNotify(player, "Flags: %s", list_flags(object->tagflags, 
		  objflaglist));

		FNotify(player, "Size: %2.1f", object->size);
	}

	if (CanSense(object))
		FNotify(player, "Sensor Range: %d", object->sensor_range);

	if (object->transporter_range > 0)	
		FNotify(player, "Transporter Range: %d", object->transporter_range);

	if (EventDriven(object))
		FNotify(player, "Critical Range: %d", object->critical_range);

	if (object->cloak_effect > 0)
		FNotify(player, "Cloak Effectiveness: %1.1f", 
		  object->cloak_effect);

	if (Ship(object)) {

		if (godmode)
			FNotify(player, "Ship Flags: %s", 
			  list_flags(ship->shipflags, shipflaglist));
		FNotify(player, "Engine output max: %d    Optimum running "
		  "level: %d%%", ship->reactor_output_max, 
		  ship->reactor_stress_level);
		FNotify(player, "Battery capacity: %d    Discharge rate: %d", 
		  ship->battery_capacity, ship->battery_discharge_max);

		if (ship->door_status != DOORS_NONE)
			FNotify(player, "Hull integrity: %d       Door size: "
			  "%2.1f", ship->hull_integrity, ship->door_size);
		else
			FNotify(player, "Hull integrity: %d", ship->hull_integrity);

		if (ship->cloak_status != CLOAK_NONE)
			FNotify(player, "Cloaking Device: Yes       Cost: %d", 
			  ship->cloak_cost);
		else
			Notify(player, "Cloaking Device: No");

		if (ship->tractor_status != TRACTOR_NONE)	
			FNotify(player, "Tractor Beam: Yes          "
			  "Effectiveness: %3.1f", ship->tractor_effect);
		else
			Notify(player, "Tractor Beam: No");

		FNotify(player, "Scanner Range: %d", ship->scanner_range);

		FNotify(player, "Maximum normal shield charge: %d", 
		  ship->shield_max[0]);

		if (ship->number_of_guns)
    			FNotify(player, "%s banks: %d   Maximum charge: %d   "
			  "Effective range: %d",
			  capstr(ship->gun_string), ship->number_of_guns,
			  ship->gun[0].power,
			  (int)(0.8 * (float)(ship->gun[0].range)));
		else
			FNotify(player, "%s banks: 0", 
			  capstr(ship->gun_string));

		if (ship->number_of_torps) 
			FNotify(player, "%s tubes: %d   Arm: %d  "
			  "Power: %d  Effective range: %d  Reloaders: %d", 
			  capstr(ship->torp_string), ship->number_of_torps,
			  ship->torp[0].power / 2, ship->torp[0].power,
			  (int)(0.6 * (float)(ship->torp[0].range)),
			  ship->num_torp_loaders);
		else
			FNotify(player, "%s tubes: 0", 
			  capstr(ship->torp_string));

	}

	return;
}
	
void load_flags(int *flags, char *flaglist)
{
	int i;

	*flags = 0;
	
	for (i=0; i < strlen(flaglist); i++)
		if (flaglist[i] != '0')	
			*flags += pow(2, i);

	return;
}

char *list_flags(int flags, SFLAGENT *list)
{
	int i;
	static char buff[SMALL_BUF_SIZE];

	strcpy(buff, "");

	for (i=0; *(list[i].name) != '\0'; i++)
		if ((flags & list[i].num) != 0)
			sprintf(buff, "%s%s ", buff, list[i].name);

	if (strlen(buff))
		buff[strlen(buff) - 1] = '\0';

	return buff;
}

void objSetObjectFlag(TAG *object, dbref player, int value, char *name)
{
	if (set_flag(&(object->tagflags), value, objflag_htab, name))
		Notify(player, "Invalid flag.");
	else {
		Notify(player, "Set.");
		if (object->space == 0)
			log_space("%s (#%d) set %s (#%d) %s%s.", Name(player), 
			  player, object->name, object->data_object, 
			  value ? "" : "!", name);
	}

	return;

}

void objSetShipFlag(TAG *object, dbref player, int value, char *name)
{
	if (set_flag(&(object->shipdata->shipflags), value, shipflag_htab,name))
		Notify(player, "Invalid flag.");
	else {
		Notify(player, "Set.");
		if (object->space == 0)
			log_space("%s (#%d) set %s (#%d) %s%s.", Name(player), 
			  player, object->name, object->data_object, 
			  value ? "" : "!", name);
	}

	return;

}

int objHasObjectFlag(TAG *object, dbref player, char *name)
{
	return (is_set(object->tagflags, objflag_htab,  name));
}

int objHasShipFlag(TAG *object, dbref player, char *name)
{
	return (is_set(object->shipdata->shipflags, shipflag_htab,  name));
}

int set_flag(int *flags, int value, HNODE *htab, char *name)
{
	SFLAGENT *flagptr;

	flagptr = (SFLAGENT *)hashSearch(name, htab);

	if (flagptr) {
		if (value)
			*flags |= flagptr->num;
		else
			*flags &= ~flagptr->num;

		return 0;
	}

	return 1;
}

int is_set(int flags, HNODE *htab, char *name)
{
	SFLAGENT *flagptr;

	flagptr = (SFLAGENT *)hashSearch(name, htab);

	if (flagptr) {

		if ((flags & flagptr->num) != 0)
			return 1;
	}

	return 0;
}	
		

/* Spacegod commands */

void objList(dbref player, int space, int show_ships, int show_other)
{

	int count;
	SPH bearing;
	TAG *ptr;

	if (space == 0)
		Notify(player, "Real Space:  ");
	else
		FNotify(player, "Sim Space %d:", space);

	Notify(player, "---------------------------------------------"
	  "-------------------------------");

	count = 0;

	for (ptr = space_list[space]; ptr != NULL; ptr=ptr->next) {

		count++;
		if (!show_ships && Ship(ptr)) continue;
		if (!show_other && !Ship(ptr)) continue;

		xyz_to_sph(ptr->pos, &bearing);

		if (!Ship(ptr) && ptr->speed==0) 
			FNotify(player,"#%-5d %-30s  %3.0f%-+3.0f %11d\t\t%c%c", 
			  ptr->data_object, ptr->name,
			  bearing.bearing, bearing.elevation, bearing.range,
			  (ptr->locked_on != NULL) ? 'L' : ' ',
			  Cloaked(ptr) ? 'C' : ' ');
		else
			FNotify(player, "#%-5d %-30s  %3.0f%-+3.0f %11d   %3.0f"
			  "%-+3.0f %4.1f\t%c%c",ptr->data_object, ptr->name, 
			  bearing.bearing, bearing.elevation, 
			  bearing.range, ptr->heading.bearing, 
			  ptr->heading.elevation, ptr->speed,
			  (ptr->locked_on != NULL) ? 'L' : ' ',
			  Cloaked(ptr) ? 'C' : ' ');

	}

	Notify(player, "---------------------------------------------"
	  "-------------------------------");
	FNotify(player, "Total objects:  %d", count);
	Notify(player, " ");

	return;
}

TAG *objFindObject(dbref target)
{
	int i;
	TAG *object;

	for (i = 0; i < NUM_SPACES; i++) { 
		for (object = space_list[i]; object != NULL; object=object->next)
			if (object->data_object == target)
				return(object);
	}

	return (NULL);

} 
