/* v0.8
 *
 * events.c:  Events code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "events.h"
#include "sensors.h"
#include "smisc.h"

char *fargs[10];

char *write_contact(CONTACT *);
void evBroadcast(TAG *, TAG *, char *, char *);

char *write_contact(CONTACT *contact)
{
	char *arg;

	arg=AllocBuffer("write_contact");

	if (contact != NULL)
		sprintf(arg, "%d", contact->contact_number);
	else
		sprintf(arg, "-1");

	return arg;
}
	
void evCheckEvents(int space)
{

	int d;
	TAG *object;
	CONTACT *contact;

	for (object = space_list[space]; object != NULL; object=object->next) {

		if (Removed(object))
			continue;

		if (!EventDriven(object))
			continue;


		for (contact = object->contact_list; contact != NULL; contact=contact->next) {
			/*
			 * Check for objects moving inside and outside
			 * critical range
			 */

			d = distance(object->pos, contact->listref->pos);

			if (contact->inside_critical && 
			  d > object->critical_range) {
				contact->inside_critical = FALSE;
				evOutsideCritical(object, contact);
			} 
			else if (!contact->inside_critical && 
			  d <= object->critical_range) {

				contact->inside_critical = TRUE;
				evInsideCritical(object, contact);
			}

		}

	}

	return;

}

void evListContainers(TAG *object, char *buff)
{
	char *ind = buff;
	TAG *ptr;
	CONTACT *con;

	strcpy(buff, "");

	for (ptr = space_list[object->space]; ptr != NULL; ptr=ptr->next) {
		for (con = ptr->contact_list; con != NULL; con=con->next) {

			if (con->listref == object && con->inside_critical) {
				sprintf(ind, "#%d ", ptr->data_object);
				ind += strlen(ind);
			}

			/* Overflow prevention */
			if (ind - buff >= LARGE_BUF_SIZE - 10)
				return;
		}
	}

	if (buff[strlen(buff) - 1] == ' ')
		buff[strlen(buff) - 1] = '\0';

	return;
}				

void evNewContact(TAG *senser, CONTACT *contact)
{

	fargs[0]=AllocBuffer("evNewContact");
	sprintf(fargs[0], "%d", contact->contact_number);

	getEvalAttr(senser->data_object, EVENT_NEW_CONTACT, fargs, 1);
}

void evCommReceived(TAG *object, TAG *sender, int freq, char *message)
{

	fargs[0]=AllocBuffer("evCommReceived");
	sprintf(fargs[0], "#%d", sender->data_object);

	fargs[1]=AllocBuffer("evCommReceived");
	sprintf(fargs[1], "%d", freq);

	fargs[2]=AllocBuffer("evCommReceived");
	strcpy(fargs[2], message);

	getEvalAttr(object->data_object, EVENT_COMM_RECEIVED, fargs, 3);
}

void evBayDoors(TAG *object, int status)
{

	fargs[0]=AllocBuffer("evBayDoors");
	sprintf(fargs[0], "%d", status);

	getEvalAttr(object->data_object, EVENT_BAY_DOORS, fargs, 1);
}

void evContactLost(TAG *senser, CONTACT *contact)
{
	fargs[0]=AllocBuffer("evContactLost");
	sprintf(fargs[0], "%d", contact->contact_number);

	getEvalAttr(senser->data_object, EVENT_CONTACT_LOST, fargs, 1);	
}

void evInsideCritical(TAG *senser, CONTACT *contact)
{

	fargs[0]=AllocBuffer("evInsideCritical");
	sprintf(fargs[0], "%d", contact->contact_number);

	getEvalAttr(senser->data_object, EVENT_INSIDE_CRITICAL, fargs, 1);

	if (senser->space == 0)
		log_space("%s (#%d) is in %s (#%d)'s critical range at range "
		  "%d.", contact->listref->name, contact->listref->data_object,
		  senser->name, senser->data_object, distance(senser->pos, 
		  contact->listref->pos));
	return;
}

void evOutsideCritical(TAG *senser, CONTACT *contact)
{

	fargs[0]=AllocBuffer("evOutsideCritical");
	sprintf(fargs[0], "%d", contact->contact_number);

	getEvalAttr(senser->data_object, EVENT_OUTSIDE_CRITICAL, fargs, 1);

	if (senser->space == 0)
		log_space("%s (#%d) is out of %s (#%d)'s critical range at "
		  "range %d.", contact->listref->name, 
		  contact->listref->data_object, senser->name, 
	  	  senser->data_object, distance(senser->pos, 
		  contact->listref->pos));

	return;
}

void evStartup(TAG *object)
{
	getEvalAttr(object->data_object, EVENT_STARTUP, (char **)NULL, 0);
}

void evShutdown(TAG *object)
{
	getEvalAttr(object->data_object, EVENT_SHUTDOWN, (char **)NULL, 0);
}

void evLockedOn(TAG *object, TAG *locker)
{
	fargs[0]=write_contact(snsFindContact(object, locker));
	getEvalAttr(object->data_object, EVENT_LOCKED_ON, fargs, 1);
}

void evLockAchieved(TAG *object)
{
	getEvalAttr(object->data_object, EVENT_LOCK_ACHIEVED, (char **)NULL, 0);
}

void evGunsCharged(TAG *object)
{
	getEvalAttr(object->data_object, EVENT_GUNS_CHARGED, (char **)NULL, 0);
}

void evTorpsArmed(TAG *object)
{
	getEvalAttr(object->data_object, EVENT_TORPS_ARMED, (char **)NULL, 0);
}

void evTorpsLoaded(TAG *object)
{
	getEvalAttr(object->data_object, EVENT_TORPS_LOADED, (char **)NULL, 0);
}

void evLockBroken(TAG *locker, TAG *lockee)
{
	getEvalAttr(locker->data_object, EVENT_LOCK_BROKEN, (char **)NULL, 0);
	fargs[0] = write_contact(snsFindContact(lockee, locker));
	getEvalAttr(lockee->data_object, EVENT_ENEMY_LOCK_BROKEN, fargs, 1);
}

void evTractorOn(TAG *object, TAG *locker)
{
	fargs[0]=write_contact(snsFindContact(object, locker));
	getEvalAttr(object->data_object, EVENT_TRACTOR_ON, fargs, 1);
	evBroadcast(object, locker, "locked a tractor beam on",
	  EVENT_TRACTOR_ON_OTHER);
}

void evTractorBroken(TAG *locker, TAG *lockee) {
	getEvalAttr(locker->data_object, EVENT_TRACTOR_BROKEN, (char **)NULL,0);
	fargs[0] = write_contact(snsFindContact(lockee, locker));
	getEvalAttr(lockee->data_object, EVENT_ENEMY_TRACTOR_BROKEN, fargs, 1);
	evBroadcast(locker, lockee, "dropped its tractor beam from",
	  EVENT_TRACTOR_BROKEN_OTHER);
}

void evBroadcast(TAG *source, TAG *object, char *msg, char *event)
{

	TAG *ptr;
	CONTACT *sourcecontact, *objcontact;
	CONTACT *contact;
	char buff[SMALL_BUF_SIZE], buff2[SMALL_BUF_SIZE];

	for (ptr = space_list[source->space]; ptr != NULL; ptr=ptr->next) {

	  if (ptr == source || ptr == object) continue;

	  sourcecontact = objcontact = NULL;

	  for (contact = ptr->contact_list; contact != NULL; contact=contact->next) {
	    if (contact->listref == source) 
		sourcecontact = contact;

	    if (contact->listref == object)
		objcontact = contact;

	  }

	  if (sourcecontact == NULL && objcontact == NULL)
		continue;
	  
	  if (Ship(ptr) && msg != NULL) {

		if (sourcecontact != NULL) 
			sprintf(buff, "Contact [%d]:  %s",
				sourcecontact->contact_number, source->name);

		else
			sprintf(buff, "An unknown source");
			
		if (object != NULL) {

			if (objcontact != NULL) {
				sprintf(buff2, " %s [%d]",
				  object->name, 
				  objcontact->contact_number);
			}
			else
				sprintf(buff2, " an unknown target");
		}
		else
			strcpy(buff2, "");

		FNotify(dbrefUser(ptr->shipdata->tactical), 
		  "%s %s%s.", buff, msg, buff2); 

	    }	   
 
	    if (EventDriven(ptr) && event != NULL) {

		fargs[0]=write_contact(sourcecontact);

		if (object != NULL) 
			fargs[1]=write_contact(objcontact);
		else {
			fargs[1]=AllocBuffer("evBroadcast");
			sprintf(fargs[1], "-1");
		}
				
		getEvalAttr(ptr->data_object, event, fargs, 2);

	    }
	}

	return;
}
			
void evLanded(TAG *lander, TAG *landee)
{
	evBroadcast(lander, landee, "has landed on", EVENT_LANDED_OTHER);

	fargs[0] = AllocBuffer("evLanded");
	sprintf(fargs[0], "#%d", landee->data_object);
	getEvalAttr(lander->data_object, EVENT_LANDED, fargs, 1);
	fargs[0] = AllocBuffer("evLanded");
	sprintf(fargs[0], "#%d", lander->data_object);
	getEvalAttr(landee->data_object, EVENT_LANDED_ON, fargs, 1);

	return;	
}

void evOrbiting(TAG *orbiter, TAG *orbitee)
{
	evBroadcast(orbiter, orbitee, "is orbiting", EVENT_ORBITING_OTHER);

	fargs[0] = AllocBuffer("evOrbiting");
	sprintf(fargs[0], "#%d", orbitee->data_object);
	getEvalAttr(orbiter->data_object, EVENT_ORBITING, fargs, 1);
	fargs[0] = AllocBuffer("evOrbiting");
	sprintf(fargs[0], "#%d", orbiter->data_object);
	getEvalAttr(orbitee->data_object, EVENT_ORBITED, fargs, 1);

	return;	
}

void evLaunched(TAG *launcher, TAG *launchee)
{
	evBroadcast(launcher, launchee, "launched from", 
	  EVENT_LAUNCHED_OTHER);

	fargs[0] = AllocBuffer("evLaunched");
	sprintf(fargs[0], "#%d", launchee->data_object);
	getEvalAttr(launcher->data_object, EVENT_LAUNCHED, fargs, 1);
	fargs[0] = AllocBuffer("evLaunched");
	sprintf(fargs[0], "#%d", launcher->data_object);
	getEvalAttr(launchee->data_object, EVENT_LAUNCHED_FROM, fargs, 1);

	return;	
}

void evDocked(TAG *docker, TAG *dockee)
{
	evBroadcast(docker, dockee, "has docked with", EVENT_DOCKED_OTHER);

	fargs[0] = AllocBuffer("evDocked");
	sprintf(fargs[0], "#%d", dockee->data_object);
	getEvalAttr(docker->data_object, EVENT_DOCKED, fargs, 1);
	fargs[0] = AllocBuffer("evDocked");
	sprintf(fargs[0], "#%d", docker->data_object);
	getEvalAttr(dockee->data_object, EVENT_DOCKED_WITH, fargs, 1);

	return;	
}

void evUndocked(TAG *undocker, TAG *undockee)
{
	evBroadcast(undocker, undockee, "has undocked from", 
	  EVENT_UNDOCKED_OTHER);

	fargs[0] = AllocBuffer("evUndocked");
	sprintf(fargs[0], "#%d", undockee->data_object);
	getEvalAttr(undocker->data_object, EVENT_UNDOCKED, fargs, 1);
	fargs[0] = AllocBuffer("evUndocked");
	sprintf(fargs[0], "#%d", undocker->data_object);
	getEvalAttr(undockee->data_object, EVENT_UNDOCKED_FROM, fargs, 1);

	return;	
}

void evCloakEngaged(TAG *object)
{
	evBroadcast(object, NULL, "is cloaking", EVENT_CLOAK_OTHER);
}	

void evCloakDisengaged(TAG *object)
{
	evBroadcast(object, NULL, "is decloaking", EVENT_DECLOAK_OTHER);
}

void evTakeDamage(TAG *target, int damage, int type)
{

	fargs[0]=AllocBuffer("evTakeDamage");
	sprintf(fargs[0], "%d", damage);
	fargs[1]=AllocBuffer("evTakeDamage");
	sprintf(fargs[1], "%d", type);

	getEvalAttr(target->data_object, EVENT_TAKE_DAMAGE, fargs, 2);

	return;
}

void evSensedExplosion(TAG *target, int damage, int range)
{
	fargs[0]=AllocBuffer("evSensedExplosion");
	sprintf(fargs[0], "%d", damage);
	fargs[1]=AllocBuffer("evSensedExplosion");
	sprintf(fargs[1], "%d", range);

	getEvalAttr(target->data_object, EVENT_EXPLOSION, fargs, 2);

	return;
}

void evExplosion(int space, XYZ location, int damage)
{

	TAG *ptr;
	XYZ rel_pos;
	SPH bearing;
	float factor;

	factor = (float) damage / 500.0;

	for (ptr = space_list[space]; ptr != NULL; ptr=ptr->next) {

		if (CanSense(ptr) && distance(location, ptr->pos) < factor * 
		  (float) ptr->sensor_range) {

			rel_pos.x = location.x - ptr->pos.x;
			rel_pos.y = location.y - ptr->pos.y;
			rel_pos.z = location.z - ptr->pos.z;
			xyz_to_sph(rel_pos, &bearing);

			if (Ship(ptr)) {
				FNotify(dbrefUser(ptr->shipdata->tactical),
				  "An explosion was detected at "
				  "%3.1f%+3.1f, range %d.", bearing.bearing,
				  bearing.elevation, bearing.range);
			}

			if (EventDriven(ptr))
				evSensedExplosion(ptr, damage, bearing.range);
		}
	}
			
	return;
}

void evReactorDrain(TAG *target)
{
	getEvalAttr(target->data_object, EVENT_REACTOR_DRAIN, (char **) NULL, 0);
}

void evDestroyed(TAG *object)
{

	getEvalAttr(object->data_object, EVENT_DESTROYED, (char **) NULL, 0);
	evBroadcast(object, NULL, "has been destroyed", EVENT_DESTROYED_OTHER);
	return;
}

void evDisabled(TAG *object)
{

	getEvalAttr(object->data_object ,EVENT_DISABLED, (char **) NULL, 0);
	evBroadcast(object, NULL, "has been disabled", EVENT_DISABLED_OTHER);
	return;
}

void evLowPower(TAG *object)
{
	getEvalAttr(object->data_object, EVENT_LOW_POWER, (char **) NULL, 0);
}

void evAttacked(TAG *object, CONTACT *attacker)
{

	if (EventDriven(object)) {
		fargs[0]=write_contact(attacker);
		getEvalAttr(object->data_object, EVENT_ATTACKED, fargs, 1);
	}

	return;
}

