/* v0.8
 *
 * eng.c:  Ship engineering code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "tactical.h"
#include "nav.h"
#include "eng.h"
#include "damage.h"
#include "smisc.h"
#include "events.h"
#include "shields.h"

void engTacticalTurn(TAG *object)
{
	char buff[SMALL_BUF_SIZE];
	SHIP *ship = object->shipdata;
	int time_left;
	int engineer = dbrefUser(ship->eng);
	int engineering = dbrefEngine(ship);
	int overload_change;

	/* Charge batteries */
	ship->battery_level += ship->alloc_batt;

	/* deplete battery */
	ship->battery_level -= ship->battery_discharge;

	/* bleed batteries */
	if (ship->battery_status == BTTY_ONLINE)
		ship->battery_level *= BATTERY_BLEED;

	/* check for bettery overcharge */
	if (ship->battery_level > ship->battery_capacity)
		ship->battery_level = ship->battery_capacity;

	/* force a reallocation if batts are too low */
	if (ship-> battery_discharge > ship->battery_level)
		engAllocCheck(object);

	/* See if we still have power to maintain life support */

	if (ship->reactor_output < (int)(0.05 * ship->reactor_output_max))
		evLowPower(object);

	if (ship->reactor_setting < ship->reactor_stress_level &&
	   ship->overload_points > 0) {

		/* we're losing overload points.  Whee! */
		ship->overload_points -= ((ship->reactor_stress_level -
		  ship->reactor_setting) / 2);

		if (ship->overload_points < 0)
			ship->overload_points = 0;
	}

	/* If there's no reactor stress, we're done. */
	if (ship->reactor_setting <= ship->reactor_stress_level)
		return;

	/* Otherwise, we're accruing overload points */
	overload_change = ship->reactor_setting - ship->reactor_stress_level;

	if (ship->reactor_setting > 100)
		overload_change += ((ship->reactor_overload_penalty) *
		  (ship->reactor_setting - 100));

	/* add them in and flash a message */
	ship->overload_points += overload_change;

	/* if reactor has overloaded too long damage may occur */
	/* call some kind of damage function here! */
	if (FRAND < ((float) ship->overload_points / ship->max_overload_points 
	  - 1) && ship->damage[SYSTEM_ENGINE].status != '9') {

		/* damage the engine 1-3 points */
		ship->damage[SYSTEM_ENGINE].status += (int)(FRAND * 2 + 1);

		if (ship->damage[SYSTEM_ENGINE].status >= '9') {

			int space, output;
			XYZ pos;
			dbref tac_officer, comm_officer, nav_officer;

			/* get the dbrefs of all the console officers */
			tac_officer = dbrefUser(ship->tactical);
			nav_officer = dbrefUser(ship->nav);
			comm_officer = dbrefUser(ship->comm);

			ship->damage[SYSTEM_ENGINE].status = '9';

			NotifyExcept(dbrefEngine(ship), GOD, NOTHING, 
			  "Antimatter chamber breach.  Ship destroyed.");
			NotifyExcept(dbrefBridge(ship), GOD, NOTHING, 
			  "Antimatter chamber breach.  Ship destroyed.");
			evDestroyed(object);

			/* If in real space, log it */
			if (object->space == 0) {

			 	log_space("%s's engine destroyed.  Tactical: %s(#%d)  "
				  "Nav: %s(#%d)  Comm: %s(#%d)  Eng: %s(#%d)", object->name, 
				  getAttrByNumber(tac_officer, A_NAME), tac_officer, 
				  getAttrByNumber(nav_officer, A_NAME), nav_officer,
				  getAttrByNumber(comm_officer, A_NAME), comm_officer, 
		    	  getAttrByNumber(engineer, A_NAME), engineer);

			}

			space = object->space;
			pos.x = object->pos.x;
			pos.y = object->pos.y;
			pos.z = object->pos.z;
			output = ship->reactor_output;

			objRemoveObject(object);
			damExplosionDamage(space, pos, output);
			return;
		}

		ship->current_reactor_output_max = ship->reactor_output_max *
		  (1-((float)(ship->damage[SYSTEM_ENGINE].status-'0') * 0.1));
		FNotify(engineer, "Reactor damaged due to overstress.  New max "
		  "power is %d.", ship->current_reactor_output_max);
		engAllocCheck(object);
	}

	/* calculate the time left until crystal stress at this pace */

	time_left = (int)(6.0 * (ship->max_overload_points -
	  ship->overload_points) / overload_change);

	if (time_left < 6 && ship->eng_safeties) {
		/* force a reactor slowdown */
		Notify(engineer, "Reactor at critical stress."
		  "  Emergency override initiated.");
		engSetReactorOutput(object, engineer, ship->reactor_stress_level);
	}
	else if (time_left < 61 && ship->eng_warnings) {

		if (time_left > 0)
			sprintf(buff, "Engine overstressing.  Reactor stress in"
				" %d seconds.", time_left);
		else
			sprintf(buff, "Engine is overstressed. Stress at %d%%.",
				100 * object->shipdata->overload_points /
				object->shipdata->max_overload_points);

		NotifyExcept(engineering, GOD, NOTHING, buff);
	}

	return;
}

void engSetReactorOutput(TAG *object, dbref player, int setting)
{

	SHIP *ship = object->shipdata;

	if (setting < 5) {
		Notify(player, "The minimum output level to maintain life "
		  "support is 5%.");
		return;
	}
	else if (setting > ship->reactor_setting_limit) {
		FNotify(player, "The maximum output level is %d%%.",
			ship->reactor_setting_limit);
		return;
	}
	else {
		ship->reactor_setting = setting;

		ship->reactor_output = (int)((ship->current_reactor_output_max
		  - ship->reactor_drain) * setting / 100);
		FNotify(player, "Reactor output level set to %d%%", setting);
		FNotify(player, "Power available: %d", ship->reactor_output);

		/* overallocation check */
		engAllocCheck(object);
		return;
	}

}

void engAllocCheck(TAG *object)
{
	SHIP *ship = object->shipdata;
	int allocation;
	int available_power;
	dbref engineer;

	engineer = dbrefUser(ship->eng);

	/* recalculate reactor_output */
	ship->reactor_output = (int)((ship->current_reactor_output_max -
	  ship->reactor_drain) * ship->reactor_setting / 100);

	available_power = ship->reactor_output;
	if (ship->battery_status == BTTY_ONLINE) {
		if (ship->battery_level < ship->battery_discharge_max)
			available_power += ship->battery_level;
		else
			available_power += ship->battery_discharge_max;
	}

	allocation = ship->alloc_tac + ship->alloc_nav + ship->alloc_shields +
	  ship->alloc_batt;

	if (allocation > available_power) {
		Notify(engineer, "Drop in net power.  Global energy "
		  "reallocation performed.");

		engAllocate(object, engineer, ship->alloc_tac, ship->alloc_nav,
		  ship->alloc_shields, ship->alloc_batt);
		navAllocCheck(object);
		shdAllocCheck(object);
	}
	else {
		if (allocation > ship->reactor_output)
			ship->battery_discharge = allocation - 
			  ship->reactor_output;
		else
			ship->battery_discharge = 0;
	}

	return;

}

void engAllocate(TAG *object, dbref player, int tac, int nav, int shields, 
  int batt)
{

	SHIP *ship = object->shipdata;
	int available_power;
	int total_allocation;
	float power_ratio;   /* fraction the the allocation to receive *
		* in the case of over-allocation.        */
	int old_nav, old_tac, old_shields;

	/* calculate max available power */
	available_power = ship->reactor_output;
	if (ship->battery_status == BTTY_ONLINE) {
		if (ship->battery_discharge_max <= ship->battery_level)
			available_power += ship->battery_discharge_max;
		else
			available_power += ship->battery_level;
	}

	/* if disabled, no power is available */
	if (Disabled(ship)) {
		Notify(player, "Ship is disabled.  Cannot allocate power.");
		available_power = 0;
	}

	/* set negative allocations to zero */
	if (tac < 0)  tac = 0;
	if (nav < 0) nav = 0;
	if (shields < 0) shields = 0;
	if (batt < 0) batt = 0;

	/* calculate total allocation */
	total_allocation = tac + nav + shields + batt;

	/* if allocation > power, apply cuts */
	if (total_allocation > available_power) {
		power_ratio = (float) ((float)available_power /
		  (float)total_allocation);
		tac = (int)((float)tac * power_ratio);
		nav = (int)((float)nav * power_ratio);
		shields = (int)((float)shields * power_ratio);
		batt = (int)((float)batt * power_ratio);
	}

	/* save the old values */
	old_nav = ship->alloc_nav;
	old_tac = ship->alloc_tac;
	old_shields = ship->alloc_shields;

	/* assign the new values */
	ship->alloc_tac = tac;
	ship->alloc_nav = nav;
	ship->alloc_shields = shields;
	ship->alloc_batt = batt;

	/* update battery discharge */
	ship->battery_discharge = tac + nav + shields + batt - ship->reactor_output;
	if (ship->battery_discharge < 0)
		ship->battery_discharge = 0;

	/* notify the player */
	FNotify(player, "Total available power: %d", available_power);
	FNotify(player, "Allocated to Tactical: %d  Nav: %d  Shields: %d  "
	  "Batteries: %d\n", tac, nav, shields, batt);

	/* check for shortfalls at other stations */
	if (old_nav > nav) {

		if (object->tractor_source != NULL)
			Notify(dbrefUser(object->tractor_source->shipdata->tactical),
			  "Tractor beam target has decreased their power to engines.");
		navAllocCheck(object);
	}

	if (old_tac > tac)
		tacAllocCheck(object);

	if (old_shields > shields)
		shdAllocCheck(object);

	/* check for a power increase at other stations */
	if (old_tac < tac) 
		FNotify(dbrefUser(ship->tactical), "New tactical allocation is %d.", 
		  tac);

	if (old_nav < nav) {

		FNotify(dbrefUser(ship->nav), "New nav allocation is %d.", nav);

		if (object->tractor_source != NULL)
			Notify(dbrefUser(object->tractor_source->shipdata->tactical),
			  "Tractor beam target has increased their power to engines.");
	}

	if (old_shields < shields) 
		FNotify(dbrefUser(ship->shields), "New shield allocation is %d.", 
		  shields);


	return;

}

void engBatteryOn(TAG *object, dbref player)
{

	SHIP *ship = object->shipdata;

	/* active.  Check battery status */
	switch (ship->battery_status) {
	  case BTTY_ONLINE:
		Notify(player, "Batteries are already online.");
		break;

	  case BTTY_DAMAGED:
		Notify(player, "Batteries are damaged and may not be brought"
		  " online.");
		break;

	  case BTTY_OFFLINE:
		ship->battery_status = BTTY_ONLINE;
		Notify(player, "Batteries now online.");
	      break;

	}

	return;

}

void engBatteryOff(TAG *object, dbref player)
{

	SHIP *ship = object->shipdata;

	switch (ship->battery_status) {
	  case BTTY_ONLINE:
		ship->battery_status = BTTY_OFFLINE;
		Notify(player, "Batteries are now offline.");

		/* check for overallocation */
		engAllocCheck(object);
		break;

	  case BTTY_DAMAGED:
		Notify(player, "Batteries are damaged and need not be ta"
		  "ken off line.");
		break;

	  case BTTY_OFFLINE:
		Notify(player, "Batteries are already offline.");
		break;
	}

	return;

}

void engShutdownReactor(TAG *object, dbref player)
{
	if (object->space == 0)	
		Notify(player, "This command is not available in real space.");

	else {
		objRemoveObject(object);
		Notify(player, "Reactor shutdown.");
	}

	return;
}

void engStartReactor(TAG *object, dbref player, dbref data)
{

	int space;

	space = atoi(getAttrByName(data, STATUS_SPACE));

	if (space == 0) {
		Notify(player, "This command is not available in real space.");
		return;
	}

	if (space_lock[space] == 1) {
		Notify(player, "Space is temporarily locked.  Please try again "
		  "later.");
		return;
	}

	objAddObject(data);

	if (objFindObject(data) == NULL) {
		Notify(player, "Unable to activate ship.  Please contact an"
		  " admin.");
		return;
	}

	Notify(player, "Reactor started and set to 5% output.");
	return;

}
		
void engSetReactor(TAG *object, dbref player, int setting)
{

	SHIP *ship = object->shipdata;

	if (setting < 5) {
		Notify(player, "The minimum output level is 5%.");
		return;
	}
	else if (setting > ship->reactor_setting_limit) {
		FNotify(player, "The maximum output level is %d%%.",
		  ship->reactor_setting_limit);
		return;
	}
	else {
		ship->reactor_setting = setting;

		ship->reactor_output = (int)(ship->current_reactor_output_max
		  * setting / 100);
		FNotify(player, "Reactor output level set to %d%%", setting);
		FNotify(player, "Power available: %d", ship->reactor_output);

		/* overallocation check */
		engAllocCheck(object);
		return;
	}
}
