/* v0.8
 *
 * comm.c:  Ship communications code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file 
 * 'COPYING' for details.
 */

#include <stdio.h>
#include <string.h>

#include "spaceconf.h" 
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "damage.h"
#include "smisc.h"
#include "comm.h"
#include "events.h"

void send_comm_message(TAG *, unsigned, char *);

void commSetChannel(TAG *object, dbref player, int chan, unsigned freq)
{
	SHIP *ship = object->shipdata;
	/* active check already done */

	/* frequency HAS to be valid since we'll accept anything that can fit
	 * in a short integer.
	 */

	if (freq == 0) {
		/* close the channel */
		ship->channel[chan].frequency = 0;
		ship->channel[chan].send_status = CS_COMM_ONLY;
		ship->channel[chan].receive_status = CR_COMM_ONLY;
		strcpy(ship->channel[chan].label, "\0");

		/* tell player about it */
		FNotify(player, "Channel %d closed.", chan);
	}
	else {
		/* set the channel up */
		ship->channel[chan].frequency = freq;
		ship->channel[chan].send_status = CS_COMM_ONLY;
		ship->channel[chan].receive_status = CR_COMM_ONLY;
		strcpy(ship->channel[chan].label, "");
   
		/* tell the player */
		FNotify(player, "Channel %d set to frequency %u.", chan, freq);
	}
	return;
}

void commLabelChannel(TAG *object, dbref player, int chan, char *label)
{
	SHIP *ship = object->shipdata;

	/* check for open channel */
	if (ship->channel[chan].frequency == 0) {
		FNotify(player, "Channel %d is closed.", chan);
		return;
	}

	/* valid channel.  Label it */
	strncpy(ship->channel[chan].label, label, 16);
	ship->channel[chan].label[16]='\0';

	/* notify the player */
	FNotify(player, "Channel %d labeled as: %s", chan, 
	  ship->channel[chan].label);

	return;

}

void commSend(TAG *object, dbref player, int chan, char *message)
{
	SHIP *ship = object->shipdata;

	/* check for open channel */
	if (ship->channel[chan].frequency == 0) {
		FNotify(player, "Channel %d is closed.", chan);
		return;
	}

	/* Hokay.  Channel is open and valid.  Check for a null message */
	if(!strlen(message)) 
		return;

	/* send the message */
	send_comm_message(object, ship->channel[chan].frequency, message);

	Notify(player, "Sent.");

	return;
}

void send_comm_message(TAG *object, unsigned freq, char *message)
{
	TAG *ptr;
	int i;

	/* 
	 * active checks were done a long time ago.  We know that we have a 
	 * valid ship, frequency, and message.  All we have to do is send it.
	 */

	for (ptr=space_list[object->space]; ptr != NULL; ptr=ptr->next) {

		/* don't send to sender */
		if (ptr == object || !Ship(ptr)) 
			continue;

		/* see if they're out of range */
		if (distance(object->pos, ptr->pos) > 
		  object->shipdata->transmitter_range * 
		  ptr->shipdata->receiver_sensitivity) 
			continue;

		/* see if they're monitoring this frequency */
		for (i = 0; i < MAX_CHANNELS; i++) {
			if (ptr->shipdata->channel[i].frequency == freq) 
				evCommReceived(ptr, object, freq, message);
		}
 	}

}
